/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.security.jndi;

import org.jboss.security.SimplePrincipal;
import org.jnp.interfaces.NamingContextFactory;

import javax.naming.Context;
import javax.naming.NamingException;
import java.security.Principal;
import java.util.Hashtable;

/** A naming provider InitialContextFactory implementation that combines the
 * authentication phase with the InitialContext creation. During the
 * getInitialContext callback from the JNDI naming, layer security context
 * identity is populated with the username obtained from the
 * Context.SECURITY_PRINCIPAL env property and the credentials from the
 * Context.SECURITY_CREDENTIALS env property. There is no actual authentication
 * of this information. It is merely made available to the jboss transport
 * layer for incorporation into subsequent invocations. Authentication and
 * authorization will occur on the server.
 *
 * @see javax.naming.spi.InitialContextFactory
 *
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public class JndiLoginInitialContextFactory extends NamingContextFactory
{
   // InitialContextFactory implementation --------------------------

   /** Take the env Context.SECURITY_PRINCIPAL and Context.SECURITY_CREDENTIALS
    * and propagate these to the SecurityAssociation principal and credential.
    * If Context.SECURITY_PRINCIPAL is a java.security.Principal then it is
    * used as is, otherwise its treated as a name using toString and a
    * SimplePrincipal is created. The Context.SECURITY_CREDENTIALS is passed
    * as is.
    * @param env
    * @throws NamingException
    */
   public Context getInitialContext(Hashtable env)
      throws NamingException
   {
      // Get the login principal and credentials from the JNDI env
      Object credentials = env.get(Context.SECURITY_CREDENTIALS);
      Object principal = env.get(Context.SECURITY_PRINCIPAL);
      Principal securityPrincipal = null;
      // See if the principal is a Principal or String
      if( principal instanceof Principal )
      {
         securityPrincipal = (Principal) principal; 
      }
      else
      {
         // Simply convert this to a name using toString
         String username = principal.toString();
         securityPrincipal = new SimplePrincipal(username);
      }
      // Associate this security context
      SecurityAssociationActions.setPrincipalInfo(securityPrincipal, credentials);
      // Now return the context using the standard jnp naming context factory
      Context iniCtx = super.getInitialContext(env);
      return iniCtx;
   }

}
