/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.detection.jndi;

import junit.framework.TestCase;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.network.NetworkRegistry;
import org.jboss.remoting.transport.Connector;
import org.jnp.server.Main;
import org.w3c.dom.Document;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.net.InetAddress;

/**
 * Just tests that detector A sees detector B when B comes online then off.
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class JNDIDetectorUnitTestCase extends TestCase
{
   // Setup for JNDI Server
   private int detectorPort = 1088;
   private String detectorHost = "localhost";
   private String contextFactory = "org.jnp.interfaces.NamingContextFactory";
   private String urlPackage = "org.jboss.naming:org.jnp.interfaces";

   public JNDIDetectorUnitTestCase(String name)
   {
      super(name);
   }

   public void testDetectors() throws Exception
   {
      detectorHost = InetAddress.getLocalHost().getHostName();

      // start JNDI server
      Main JNDIServer = new Main();
      JNDIServer.setPort(detectorPort);
      JNDIServer.setBindAddress(detectorHost);
      JNDIServer.start();

      Thread.currentThread().sleep(4000);
      
      JNDIDetector detector1 = new JNDIDetector();
      JNDIDetector detector2 = new JNDIDetector();

      Connector connector1 = new Connector();
      Connector connector2 = new Connector();

      NetworkRegistry reg1 = setupServers(detector1, connector1);
      NetworkRegistry reg2 = setupServers(detector2, connector2);

      // Need to allow heartbeat so have detection
      Thread.currentThread().sleep(4000);

      //Should now have an entry for both of the registries
      int reg1Count = reg1.getServers().length;
      int reg2Count = reg2.getServers().length;

      // Actual junit test
      assertTrue(reg1Count == 1 && reg2Count == 1);

      if(reg1Count == 1 && reg2Count == 1)
      {
         System.out.println("PASSED - both registries have found the detectors.");
      }
      else
      {
         System.out.println("FAILED - registries not populated with remote detectors.");
      }

      // stop the 2nd detector, so see if 1st one detects it is missing
      connector1.stop();
      //connector1.destroy();
      connector1 = null;
      connector2.stop();
      connector2.stop();
      connector2 = null;
      detector1.stop();

      // sleep for a few seconds so the 1st detector can discover 2nd one down
      Thread.currentThread().sleep(6000);

      // 1st one should be empty
      reg1Count = reg2.getServers().length;

      // Actual junit test
      assertTrue(reg1Count == 0);

      if(reg1Count == 0)
      {
         System.out.println("PASSED - 2nd detector stopped and no longer in registry.");
      }
      else
      {
         System.out.println("FAILED - 2nd detector stopped but still in registry.");
      }

      // cleanup
      detector2.stop();
      //connector2.stop();
      //connector2.destroy();
      JNDIServer.stop();
   }


   private synchronized NetworkRegistry setupServers(JNDIDetector detector, Connector connector)
   {
      NetworkRegistry registry = null;
      System.setProperty("jboss.identity", String.valueOf(System.currentTimeMillis()));
      System.out.println("jboss.identity = " + System.getProperty("jboss.identity"));

      try
      {
         MBeanServer server = MBeanServerFactory.createMBeanServer();

         //registry = NetworkRegistry.getInstance();
         registry = TestNetworkRegistry.createNetworkRegistry();
         server.registerMBean(registry, new ObjectName("remoting:type=NetworkRegistry"));

         //int port = Math.abs(new Random().nextInt(2000));
         int port = 8085;
         System.out.println("port = " + port);

         InvokerLocator locator = new InvokerLocator("socket://localhost:" + port);
         StringBuffer buf = new StringBuffer();
         buf.append("<?xml version=\"1.0\"?>\n");
         buf.append("<handlers>\n");
         buf.append("  <handler subsystem=\"mock\">org.jboss.remoting.transport.mock.MockServerInvocationHandler</handler>\n");
         buf.append("</handlers>\n");
         Document xml = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new ByteArrayInputStream(buf.toString().getBytes()));
         connector.setInvokerLocator(locator.getLocatorURI());
         connector.setConfiguration(xml.getDocumentElement());
         ObjectName obj = new ObjectName("jboss.remoting:type=Connector,transport=" + locator.getProtocol());
         server.registerMBean(connector, obj);
         //connector.create();
         connector.start();

         //Need to set new domain for identity
         server.registerMBean(detector, new ObjectName("remoting:type=JNDIDetector"));

         // set config info for detector and start it.
         detector.setPort(detectorPort);
         detector.setHost(detectorHost);
         detector.setContextFactory(contextFactory);
         detector.setURLPackage(urlPackage);
         detector.start();
      }
      catch(Exception e)
      {
         e.printStackTrace();
      }

      return registry;
   }

   private static class TestNetworkRegistry extends NetworkRegistry
   {
      public static NetworkRegistry createNetworkRegistry()
      {
         return new TestNetworkRegistry();
      }
   }

}
