/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.marshal.serializable;

import org.jboss.remoting.loading.ObjectInputStreamWithClassLoader;
import org.jboss.remoting.marshal.UnMarshaller;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.Map;

/**
 * Will perform the deserialization of objects off the wire.
 *
 * @author <a href="mailto:tom@jboss.org">Tom Elrod</a>
 */
public class SerializableUnMarshaller implements UnMarshaller
{
   public final static String DATATYPE = "serializable";

   private ClassLoader customClassLoader = null;

   /**
    * Reads the data from the input stream and converts to an Object.
    * <p/>
    * If the inputStream passed is an ObjectInputStream (which would prefer it not be), it will just
    * use it as given.  If the input stream is not an instance of ObjectInputStream, will wrap it with a
    * custom ObjectInputStream (ObjectInputStreamWithClassLoader) and use it.  The ObjectInputStreamWithClassLoader
    * will use the custom class loader set in order to ensure that any classes not found within the local classloader
    * can be loaded from the server and used within the client VM.  If the inpustream is of type ObjectInputStreamWithClassLoader,
    * then will just set the classloader to the custom classloader and proceed.<p>
    *
    * @param inputStream
    * @param metadata
    * @return
    * @throws IOException
    * @throws ClassNotFoundException
    */
   public Object read(InputStream inputStream, Map metadata) throws IOException, ClassNotFoundException
   {
      ObjectInputStream objInputStream = null;
      Object obj = null;
      if(inputStream instanceof ObjectInputStreamWithClassLoader)
      {
         if(((ObjectInputStreamWithClassLoader) inputStream).getClassLoader() == null)
         {
            ((ObjectInputStreamWithClassLoader) inputStream).setClassLoader(customClassLoader);
         }
         objInputStream = (ObjectInputStream) inputStream;
      }
      else if(inputStream instanceof ObjectInputStream)
      {
         objInputStream = (ObjectInputStream) inputStream;
      }
      else
      {
         if(customClassLoader != null)
         {
            objInputStream = new ObjectInputStreamWithClassLoader(inputStream, customClassLoader);
         }
         else
         {
            objInputStream = new ObjectInputStream(inputStream);
         }
      }

      obj = objInputStream.readObject();

      objInputStream.readObject(); // for stupid ObjectInputStream reset
      return obj;
   }

   /**
    * Sets the classloader to be used when deserializing objects off the wire.  This will ONLY be used in the
    * when the input stream passed to the read() method is NOT an instance of ObjectInputStream.<p>
    *
    * @param classloader
    */
   public void setClassLoader(ClassLoader classloader)
   {
      this.customClassLoader = classloader;
   }

}