/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package javax.emb;

import javax.ejb.EJBLocalObject;

/**
 * This basic interface defines behavior common to all kinds of metadata
 * objects that are persistent and can be altered. It also extends
 * EJBLocalObject and therefore represents the local interface of a Metadata
 * Entity EJB.
 * 
 * <p>MetaDataEntity EJBs consist of the two basic alterable properties:
 * 
 * <ul><li>The xml property models the XML content of a Meta Data Entity EJB.
 * XML content may be implicitly processed for the creation of persistent
 * indexing information to allow efficient content search. The default for this
 * property is <code>null</code>.
 * 
 * <li>The name property models a file name that is used as a default in case
 * the xml content has to be exported to a file. The default value for this
 * property is <code>null</code>.
 * 
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo
 *         Argello</a>
 */
public interface MetaDataEntityLocal extends EJBLocalObject
{
   /**
    * Adds the given Metadata Entity EJB to the receiver's set of children. If
    * the given MetaDataEntity EJB is already part of the receiver's list of
    * children no action is performed.
    * 
    * @param child
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    */
   void addChild(MetaDataEntityLocal child) throws MediaException;

   /**
    * Adds the given Media Entity EJB to the receiver's set of associated Media
    * Entity EJBs that are described by the receiver. If the given Media Entity
    * EJB is already part of the receiver's list no action is performed.
    * 
    * @param mediaEntity
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    */
   void addMediaEntity(MediaEntityLocal mediaEntity) throws MediaException;

   // This method is listed in the code snippet at page 55 in the
   // Proposed Final Draft, but I can't find it's description.
   // URL exportMedia(URL) throws MediaException;

   /**
    * Returns the receiver's children as an array of Media Entity EJBs. The
    * array is empty if no children are related.
    */
   MetaDataEntityLocal[] getChildren() throws MediaException;

   /**
    * Returns a timestamp stating when the receiver's persistent state was last
    * modified.
    * 
    * @return @throws MediaException
    */
   long getLastModified() throws MediaException;

   /**
    * Returns the Media Entity EJBs associated with the receiver. The array is
    * empty if no media entity EJBs are related.
    * 
    * @return @throws MediaException
    */
   MediaEntityLocal[] getMediaEntities() throws MediaException;

   /**
    * This method locates all Media Entity EJBs associated with the receiver
    * that have the specified media format. If the given recursion flag is
    * true, the search is extended to all Media Entity EJBs that relate to the
    * receiver or one of its recursive children.
    * 
    * @return MediaEntityLocal array.
    * @throws java.lang.NullPointerException if the media format passed is
    *         <code>null</code>.
    */
   MediaEntityLocal[] getMediaEntities(
      MediaFormat mediaFormat,
      boolean searchCildren)
      throws MediaException;

   /**
    * This method locates all Media Entity EJBs associated with the receiver
    * that have the given mime type. If the given recursion flag is <code>true</code>,
    * the search is extended to all Media Entity EJBs that relate to the
    * receiver or one of its recursive children.
    * 
    * @return MediaEntityLocal array.
    * @throws java.lang.NullPointerException if the mime type passed is <code>null</code>.
    */
   MediaEntityLocal[] getMediaEntities(String mimeType, boolean searchCildren)
      throws MediaException;

   /**
    * Returns the receiver's non-unique name as a String. The name is used as a
    * file name hint in case the metadata XML content is to be stored in a file
    * system and therefore may only contain characters that are valid in file
    * names. Also, it should contain a file extension that represents the
    * receiver's format. Note that as opposed to media entity EJBs, the name
    * property is not required to contain a file extension registered with the
    * MediaFormatRegistry.
    * 
    * @return name.
    * @throws MediaException
    */
   String getName() throws MediaException;

   /**
    * Returns the succeeding version edition of the receiver, which allows
    * querying and a history chain of metadata objects that represent the same
    * thing. The value <code>null</code> is returned if no next version
    * exists.
    * 
    * @return media entity.
    * @throws MediaException
    */
   MetaDataEntityLocal getNextVersion() throws MediaException;

   /**
    * Returns the receiver's parents as an array of Metadata Entity EJBs. The
    * array is empty if no parents are related.
    * 
    * @return MediaEntityLocal array.
    * @throws MediaException
    */
   MetaDataEntityLocal[] getParents() throws MediaException;

   /**
    * Returns the previous version of the receiver, which allows querying a
    * history of metadata objects that represent the same logical thing. The
    * value <code>null</code> is returned if no previous version exists.
    * 
    * @return media entity.
    * @throws MediaException
    */
   MetaDataEntityLocal getPreviousVersion() throws MediaException;

   /**
    * This method returns the receiver's content as an XML string.
    * 
    * @return content as an XML string.
    * @throws javax.emb.ContentAccessException if the value passed is <code>null</code>.
    */
   String getXML() throws MediaException;

   /**
    * Removes the given Metadata Entity EJB from the receiver's set of
    * children. If the given MetaDataEntity EJB is not part of the receiver's
    * list of children, no action is performed.
    * 
    * @param child
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    * @throws MediaException
    */
   void removeChild(MetaDataEntityLocal child) throws MediaException;

   /**
    * Removes the given Media Entity EJB from the receiver's set of associated
    * Media Entity EJBs that are described by the receiver. If the given Media
    * Entity EJB is not part of the receiver's list no action is performed.
    * 
    * @param mediaEntity
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    * @throws MediaException
    */
   void removeMediaEntity(MediaEntityLocal mediaEntity) throws MediaException;

   /**
    * Sets the receiver's non-unique name as a String. The name is used as a
    * file name hint in case the media content is to be stored or published in
    * a file system and therefore may only contain characters that are valid in
    * file names. Also, it should contain a file extension that represents the
    * receiver's media format. Note that as opposed media entity EJBs the name
    * property is not required to contain a file extension registered with the
    * MediaFormatRegistry.
    * 
    * @param name
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    * @throws MediaException
    */
   void setName(String name) throws MediaException;

   /**
    * Defines the given metadata entity to be the previous version of the
    * receiver, which allows querying a history chain of metadata objects that
    * represent the same logical thing. In return, the operation causes the
    * receiver to be the given metadata entity's successor. Passing the value
    * <code>null</code> causes the receiver not to have a predecessor
    * anymore. The operation is only allowed if version chain integrity is
    * preserved:
    * 
    * <ul><li>If the given metadata entity EJB is the receiver itself: A
    * javax.emb.VersionChainIntegrityException is thrown.</li>
    * 
    * <li>If the given metadata entity EJB is already the previous version of
    * the receiver: No action is performed.</li>
    * 
    * <li>If the given metadata entity EJB is <code>null</code>: A
    * javax.emb.VersionChainIntegrityException is thrown if the receiver has a
    * successor.</li>
    * 
    * <li>Otherwise: A javax.emb.VersionChainIntegrityException is thrown if
    * the given metadata entity EJB has a successor, or if the receiver has a
    * predecessor, a successor, or both.</li></ul>
    * 
    * @param metadata
    * @throws MediaException
    */
   void setPreviousVersion(MetaDataEntityLocal metadata) throws MediaException;

   /**
    * If the given XML content is well formed, it replaces the receiver's
    * current metadata content. If the given validation flag is <code>true</code>,
    * the content is additionally strictly validated before the operation is
    * performed.
    * 
    * @param xmlContent
    * @param validate
    * @throws java.lang.NullPointerException if the content passed is <code>null</code>.
    * @throws javax.emb.MetaDataValidationException if the validate flag is
    *         <code>true</code> and the content validation fails.
    * @throws javax.emb.MetaDataSyntaxException if the XML content is not well
    *         formed, regardless of the validate flag.
    */
   void setXML(String xmlContent, boolean validate) throws MediaException;
}