/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package javax.emb;

import java.net.URL;

/**
 * This class models a portion of a (usually non-embedded) medium that is
 * optionally followed by a reference to an external media file represented by
 * a media location URL.
 * 
 * <p>Splitting up media content into media segments is useful if references
 * contained have to be updated, for example because one of the children has
 * moved. In order to handle such a situation it is necessary to disassemble
 * the media, update the reference, and reassemble it afterwards. MediaFormat
 * instances provide the operations necessary, and the MediaSegment class
 * describes the data container for a single segment.
 * 
 * <p>The value <code>null</code> is a valid value for the childLocation
 * property and indicates that the segment content is not followed by any
 * reference, for example because the end of media is reached. The content of
 * media segments is restricted to a theoretical maximum size of 2GB due to the
 * Java restriction of byte array size. However this should not impose a
 * restriction in practice as non-embedded media tends to be fairly small in
 * size, and embedded media doesn't require segmentation at all.
 * 
 * <p>Media segments also implement {@link java.io.Serializable} in order to
 * allow instances to be exchanged over remote boundaries.
 * 
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo
 *         Argello</a>
 */
public final class MediaSegment
{
   private byte[] content;
   private URL childLocation;

   /**
    * Default constructor for media segments. The content field is initialized
    * to an empty byte array, and the child location field is initialized to
    * <code>null</code>.
    */
   public MediaSegment()
   {
      content = new byte[0];
      childLocation = null;
   }

   /**
    * Returns the content of the media segment as a byte array.
    * 
    * @return the segment content.
    */
   public byte[] getContent()
   {
      return content;
   }

   /**
    * Returns the media location of a media resource referenced by the
    * receiver.
    * 
    * @return the child location.
    */
   public URL getChildLocation()
   {
      return childLocation;
   }

   /**
    * Sets the content of the media segment.
    * 
    * @param content the segment content.
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    */
   public void setContent(byte[] content)
   {
      if (content == null)
      {
         throw new NullPointerException();
      }

      this.content = content;
   }

   /**
    * Sets the media location of a media resource referenced by the receiver.
    * Passing the value <code>null</code> is allowed.
    * 
    * @param childLocation the child location.
    */
   public void setChildLocation(URL childLocation)
   {
      this.childLocation = childLocation;
   }
}