/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package test.compliance.server;

import junit.framework.TestCase;

import javax.management.MBeanServer;
import javax.management.MBeanServerBuilder;
import javax.management.MBeanServerDelegate;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;

import test.compliance.server.support.BuilderTest;
import test.compliance.server.support.DelegateListener;
import test.compliance.server.support.MBeanServerReplaced;
import test.compliance.server.support.MBeanServerWrapper;
import test.compliance.server.support.TestMBeanServerBuilder;
import test.compliance.server.support.TestMBeanServerDelegate;

/**
 * Tests the MBeanServerBuilder
 *
 * @author  <a href="mailto:Adrian.Brock@HappeningTimes.com">Adrian Brock</a>.
 */
public class MBeanServerBuilderTestCase
  extends TestCase
{
   // Attributes ----------------------------------------------------------------

   private ObjectName name;
   private ObjectName delegateName;
   private String defaultBuilder;

   private static final String DEFAULT_DOMAIN = "default";
   private static final String BUILDER_PROPERTY = "javax.management.builder.initial";

   // Constructor ---------------------------------------------------------------

   /**
    * Construct the test
    */
   public MBeanServerBuilderTestCase(String s)
   {
      super(s);
      try
      {
         name = new ObjectName("server:test=MBeanServerBuilder");
         delegateName = new ObjectName("JMImplementation:type=MBeanServerDelegate");
      }
      catch (Exception e)
      {
         e.printStackTrace();
         throw new RuntimeException(e.toString());
      }

      defaultBuilder = System.getProperty(BUILDER_PROPERTY, "javax.management.MBeanServerBuilder");
   }

   // Tests ---------------------------------------------------------------------

   public void testNewMBeanServerDelegate()
      throws Exception
   {
      MBeanServerBuilder builder = new MBeanServerBuilder();
      MBeanServerDelegate delegate = builder.newMBeanServerDelegate();
   }

   public void testNewMBeanServer()
      throws Exception
   {
      MBeanServerBuilder builder = new MBeanServerBuilder();
      MBeanServerDelegate delegate = builder.newMBeanServerDelegate();
      MBeanServer server = builder.newMBeanServer(DEFAULT_DOMAIN, null, delegate);
      assertTrue(server.getDefaultDomain().equals(DEFAULT_DOMAIN));
   }

   public void testNewMBeanServerViaMBeanServerFactory()
      throws Exception
   {
      MBeanServer server = MBeanServerFactory.newMBeanServer(DEFAULT_DOMAIN);
      assertTrue(server.getDefaultDomain().equals(DEFAULT_DOMAIN));
   }

   public void testPreRegisterGetsTheOriginalMBeanServer()
      throws Exception
   {
      MBeanServer server = MBeanServerFactory.newMBeanServer(DEFAULT_DOMAIN);

      BuilderTest test = new BuilderTest();
      server.registerMBean(test, name);

      assertTrue("Registered", server.isRegistered(name));
      assertTrue("Got the correct MBeanServer", test.server.equals(server));
   }

   public void testPreRegisterGetsTheOuterMBeanServer()
      throws Exception
   {
      MBeanServerBuilder builder = new MBeanServerBuilder();
      MBeanServerDelegate delegate = builder.newMBeanServerDelegate();
      MBeanServer wrapper = MBeanServerWrapper.getWrapper();
      MBeanServer server = builder.newMBeanServer(DEFAULT_DOMAIN, wrapper, delegate);

      MBeanServerWrapper.getHandler(wrapper).server = server;

      BuilderTest test = new BuilderTest();
      wrapper.registerMBean(test, name);

      assertTrue("Registered", server.isRegistered(name));
      assertTrue("Got the correct MBeanServer", test.server.equals(wrapper));
      assertTrue("Wrapper invoked", MBeanServerWrapper.getHandler(wrapper).invoked);
   }

   public void testMBeanServerDelegateReplaced()
      throws Exception
   {
      MBeanServerBuilder builder = new MBeanServerBuilder();
      TestMBeanServerDelegate delegate = new TestMBeanServerDelegate();
      MBeanServer server = builder.newMBeanServer(DEFAULT_DOMAIN, null, delegate);

      server.getAttribute(delegateName, "MBeanServerId");

      assertTrue("Delegate replaced", delegate.invoked);
   }

   public void testReplaceMBeanServerBuilderMBeanServerDelegate()
      throws Exception
   {
      System.setProperty(BUILDER_PROPERTY, TestMBeanServerBuilder.class.getName());
      try
      {
         MBeanServer server = MBeanServerFactory.newMBeanServer();

         DelegateListener listener = new DelegateListener();
         server.addNotificationListener(delegateName, listener, null, null);

         BuilderTest test = new BuilderTest();
         server.registerMBean(test, name);

         assertTrue("Delegate replaced", listener.userData.equals("replaced"));
      }
      finally
      {
         System.setProperty(BUILDER_PROPERTY, defaultBuilder);
      }
   }

   public void testReplaceMBeanServerBuilderMBeanServer()
      throws Exception
   {
      System.setProperty(BUILDER_PROPERTY, TestMBeanServerBuilder.class.getName());
      try
      {
         MBeanServer server = MBeanServerFactory.newMBeanServer();

         boolean caught = false;
         try
         {
            server.queryMBeans(null, null);
         }
         catch (MBeanServerReplaced e)
         {
            caught = true;
         }

         assertTrue("MBeanServer replaced", caught);
      }
      finally
      {
         System.setProperty(BUILDER_PROPERTY, defaultBuilder);
      }
   }
}
