/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.mx.notification;

import javax.management.Notification;
import javax.management.NotificationListener;

import org.jboss.logging.Logger;
import org.jboss.mx.util.JBossNotificationBroadcasterSupport;
import org.jboss.util.threadpool.BasicThreadPool;
import org.jboss.util.threadpool.ThreadPool;

/**
 * A notification broadcaster with asynch notifications
 * 
 * @author  <a href="mailto:Adrian.Brock@HappeningTimes.com">Adrian Brock</a>.
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public class AsynchNotificationBroadcasterSupport
   extends JBossNotificationBroadcasterSupport
{
   // Attributes ----------------------------------------------------
   private static Logger log = Logger.getLogger(AsynchNotifier.class);
   /** The default pool used in the absence of on instance specific one */
   private static ThreadPool defaultPool = new BasicThreadPool("AsynchNotificationBroadcasterSupport");
   /** The default comp*/
   private static long defaultNotificationTimeout;

   /** The instance */
   private long notificationTimeout;
   private ThreadPool pool;

   public static synchronized void setDefaultThreadPool(ThreadPool tp)
   {
      defaultPool = tp;
   }

   public static long getDefaultNotificationTimeout()
   {
      return defaultNotificationTimeout;
   }
   public static void setDefaultNotificationTimeout(long defaultNotificationTimeout)
   {
      AsynchNotificationBroadcasterSupport.defaultNotificationTimeout = defaultNotificationTimeout;
   }

   // Constructor ---------------------------------------------------

   /**
    * Construct a new Asyncrhonous broadcaster
    * Calls this(defaultNotificationTimeout, defaultPool)
    */
   public AsynchNotificationBroadcasterSupport()
   {
      this(defaultNotificationTimeout, defaultPool);
   }
   /**
    * Construct a new Asyncrhonous broadcaster. Calls
    * this(notificationTimeout, defaultPool)
    * @param notificationTimeout the notification completion timeout in MS. A
    * 0 value means no timeout.
    */ 
   public AsynchNotificationBroadcasterSupport(long notificationTimeout)
   {
      this(notificationTimeout, defaultPool);
   }
   /**
    * Construct a new Asyncrhonous broadcaster
    * @param notificationTimeout - the notification completion timeout in MS. A
    * 0 value means no timeout.
    * @param pool - the thread pool to use for the asynchronous notifcations
    */ 
   public AsynchNotificationBroadcasterSupport(long notificationTimeout,
      ThreadPool pool)
   {
      this.notificationTimeout = notificationTimeout;
      this.pool = pool;
   }

   // Public --------------------------------------------------------

   public long getNotificationTimeout()
   {
      return notificationTimeout;
   }
   public void setNotificationTimeout(long notificationTimeout)
   {
      this.notificationTimeout = notificationTimeout;
   }

   public ThreadPool getThreadPool()
   {
      return pool;
   }
   public void setThreadPool(ThreadPool pool)
   {
      this.pool = pool;
   }

   // NotificationBroadcasterSupport overrides ----------------------

   /**
    * Handle the notification, asynchronously invoke the listener.
    *
    * @param listener the listener to notify
    * @param notification the notification
    * @param handback the handback object
    */
   public void handleNotification(NotificationListener listener,
                                     Notification notification,
                                     Object handback)
   {
      AsynchNotifier notifier = new AsynchNotifier(listener, notification, handback);
      pool.run(notifier, 0, notificationTimeout);
   }

   /** Invoke stop on the thread pool if its not the class default pool.
    * 
    * @param immeadiate the immeadiate flag passed to the TheadPool#stop
    */ 
   protected void stopThreadPool(boolean immeadiate)
   {
      if( pool != defaultPool )
      {
         pool.stop(immeadiate);
      }
   }

   // Inner classes -------------------------------------------------

   public class AsynchNotifier
      implements Runnable
   {
      NotificationListener listener;
      Notification notification;
      Object handback;
      public AsynchNotifier(NotificationListener listener,
                            Notification notification,
                            Object handback)
      {
         this.listener = listener;
         this.notification = notification;
         this.handback = handback;
      }

      public void run()
      {
         try
         {
            listener.handleNotification(notification, handback);
         }
         catch (Throwable throwable)
         {
            log.error("Error processing notification=" + notification +
                      " listener=" + listener +
                      " handback=" + handback,
                      throwable);
         }
      }
   }
}
