/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.console.manager;

import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;

import javax.management.ObjectName;
import javax.management.MBeanServer;
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.IOException;
import java.net.URL;

/**
 * This class wraps the file system
 * for deployments.  It gives a file-based
 * persistence mechanism for deployments.
 * Used by web-console to store -service.xml files,
 * -ds.xml files, etc..., really anything text based.
 *
 * Deployments are tied to a specific name and that name
 * corresponds to the base file name.
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @version $Revision:1$
 *
 **/
public class DeploymentFileRepository extends ServiceMBeanSupport implements DeploymentFileRepositoryMBean
{
   private String baseDir;
   private File base;
   /** The server's home directory, for relative paths. */
   protected File serverHome;
   protected URL serverHomeURL;

   /**
    *
    * @param folder relative directory
    * @param name  base name of file.  Whitespace will be removed from name and replaced with '_'
    * @param fileExtension must have a '.' in ext
    * @param data
    * @param noHotDeploy  keep timestamp of file so it doesn't do a redeploy
    * @throws IOException
    */
   public void store(String folder, String name, String fileExtension, String data, boolean noHotDeploy) throws IOException
   {
      log.debug("store called");
      File dir = new File(base, folder);
      log.debug("respository folder: " + dir.toString());
      log.debug("absolute: " + dir.getAbsolutePath());
      if (!dir.exists())
      {
         if (!dir.mkdirs())
         {
            throw new RuntimeException("Failed to create directory: " + dir.toString());
         }
      }
      String filename = name.replace(' ', '_') + fileExtension;
      File file = new File(dir, filename);
      File tmpfile = new File(dir, filename + ".tmp");
      PrintWriter writer = new PrintWriter(new FileOutputStream(tmpfile));
      writer.write(data);
      writer.close();
      if (file.exists() && noHotDeploy)
      {
         long modified = file.lastModified();
         tmpfile.setLastModified(modified);
         file.delete();
      }
      if (!tmpfile.renameTo(file))
      {
         throw new RuntimeException("Failed to rename tmpfile to " + file.toString());
      }
   }

   public void remove(String folder, String name, String fileExtension)
   {
      File dir = new File(base, folder);
      String filename = name.replace(' ', '_') + fileExtension;
      File file = new File(dir, filename);
      file.delete();
   }

   public boolean isStored(String folder, String name, String fileExtension)
   {
      File dir = new File(base, folder);
      String filename = name.replace(' ', '_') + fileExtension;
      File file = new File(dir, filename);
      return file.exists();
   }

   public String getBaseDir()
   {
      return baseDir;
   }

   public void setBaseDir(String baseDir)
   {
      this.baseDir = baseDir;
      this.base = new File(serverHome, baseDir);
   }


   public ObjectName preRegister(MBeanServer server, ObjectName name)
      throws Exception
   {
      // get server's home for relative paths, need this for setting
      // attribute final values, so we need to do it here
      ServerConfig serverConfig = ServerConfigLocator.locate();
      serverHome = serverConfig.getServerHomeDir();
      return super.preRegister(server, name);
   }

}
