/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 *
 */
package org.jboss.resource.connectionmanager;

import java.util.ArrayList;

import javax.transaction.RollbackException;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.jboss.logging.Logger;
import org.jboss.tm.TransactionLocal;

/**
 * Organizes transaction synchronization done by JCA.<p>
 * 
 * This class exists to make sure all TxRemover synchronizations
 * are invoked before the cached connection manager closes any
 * closed connections.
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @version $Revision:1$
 */
public class TransactionSynchronizer implements Synchronization
{
   // Constants -----------------------------------------------------

   /** The logger */
   private static final Logger log = Logger.getLogger(TransactionSynchronizer.class);
   
   // Attributes ----------------------------------------------------

   /** The transaction synchronizations */
   protected static TransactionLocal txSynchs;
   
   /** The transaction */
   protected Transaction tx;
   
   /** TxRemover synchronizations */
   protected ArrayList txRemoverSynchs;
   
   /** The cached connection manager synchronization */
   protected Synchronization ccmSynch;
   
   // Static --------------------------------------------------------

   /** Initialization */
   public static void setTransactionManager(TransactionManager tm)
   {
      txSynchs = new TransactionLocal(tm);
   }
   
   /**
    * Register a new TxRemover synchronization
    * 
    * @param tx the transaction
    * @param synch the synchronization
    * @throws RolledbackException if the transaction is already rolled back
    * @throws SystemException for an error in the tranaction manager
    */
   public static void registerTxRemoverSynchronization(Transaction tx, Synchronization synch) throws RollbackException, SystemException
   {
      TransactionSynchronizer ts = getRegisteredSynchronizer(tx);
      if (ts.txRemoverSynchs == null)
         ts.txRemoverSynchs = new ArrayList();
      ts.txRemoverSynchs.add(synch);
   }
   
   /**
    * Check whether we have a CCM synchronization
    * 
    * @param tx the transaction
    */
   public static Synchronization getCCMSynchronization(Transaction tx)
   {
      TransactionSynchronizer ts = (TransactionSynchronizer) txSynchs.get(tx);
      if (ts != null)
         return ts.ccmSynch;
      else
         return null;
   }
   
   /**
    * Register a new CCM synchronization
    * 
    * @param tx the transaction
    * @param synch the synchronization
    * @throws RolledbackException if the transaction is already rolled back
    * @throws SystemException for an error in the tranaction manager
    */
   public static void registerCCMSynchronization(Transaction tx, Synchronization synch) throws RollbackException, SystemException
   {
      TransactionSynchronizer ts = getRegisteredSynchronizer(tx);
      ts.ccmSynch = synch;
   }
   
   // Constructors --------------------------------------------------
   
   /**
    * Create a new transaction synchronizer
    * 
    * @param tx the transaction to synchronize with
    */
   private TransactionSynchronizer(Transaction tx)
   {
      this.tx = tx;
   }

   // Public --------------------------------------------------------
   
   // Synchronization implementation --------------------------------

   public void beforeCompletion()
   {
      if (txRemoverSynchs != null)
      {
         for (int i = 0; i < txRemoverSynchs.size(); ++i)
         {
            Synchronization synch = (Synchronization) txRemoverSynchs.get(i);
            invokeBefore(synch);
         }
      }
      
      if (ccmSynch != null)
         invokeBefore(ccmSynch);
   }

   public void afterCompletion(int status)
   {
      if (txRemoverSynchs != null)
      {
         for (int i = 0; i < txRemoverSynchs.size(); ++i)
         {
            Synchronization synch = (Synchronization) txRemoverSynchs.get(i);
            invokeAfter(synch, status);
         }
      }
      
      if (ccmSynch != null)
         invokeAfter(ccmSynch, status);
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   /**
    * Invoke a beforeCompletion
    * 
    * @param synch the synchronization
    */
   protected void invokeBefore(Synchronization synch)
   {
      try
      {
         synch.beforeCompletion();
      }
      catch (Throwable t)
      {
         log.warn("Transaction " + tx + " error in before completion " + synch, t);
      }
   }

   /**
    * Invoke an afterCompletion
    * 
    * @param synch the synchronization
    * @param status the status of the transaction
    */
   protected void invokeAfter(Synchronization synch, int status)
   {
      try
      {
         synch.afterCompletion(status);
      }
      catch (Throwable t)
      {
         log.warn("Transaction " + tx + " error in after completion " + synch, t);
      }
   }
   
   /**
    * Get a registered transaction synchronizer.
    *
    * @param tx the transaction
    * @return the registered transaction synchronizer for this transaction
    * @throws RolledbackException if the transaction is already rolled back
    * @throws SystemException for an error in the tranaction manager
    */
   protected static TransactionSynchronizer getRegisteredSynchronizer(Transaction tx) throws RollbackException, SystemException
   {
      TransactionSynchronizer result = (TransactionSynchronizer) txSynchs.get(tx);
      if (result == null)
      {
         result = new TransactionSynchronizer(tx);
         tx.registerSynchronization(result);
         txSynchs.set(tx, result);
      }
      return result;
   }
   
   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
