/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.resource.adapter.jdbc.local;

import org.jboss.resource.JBossResourceException;

import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.ResourceException;
import javax.security.auth.Subject;
import java.util.List;
import java.util.ArrayList;
import java.util.Properties;
import java.util.Collections;
import java.sql.Driver;
import java.sql.Connection;

/**
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 * @version <tt>$Revision:1$</tt>
 */
public class HALocalManagedConnectionFactory
   extends LocalManagedConnectionFactory
{
   private URLSelector urlSelector;
   private String urlDelimeter;

   public String getURLDelimeter()
   {
      return urlDelimeter;
   }

   public void setURLDelimeter(String urlDelimeter)
   {
      this.urlDelimeter = urlDelimeter;
      if(getConnectionURL() != null)
      {
         initUrlSelector();
      }
   }

   public void setConnectionURL(String connectionURL)
   {
      super.setConnectionURL(connectionURL);
      if(urlDelimeter != null)
      {
         initUrlSelector();
      }
   }

   public ManagedConnection createManagedConnection(Subject subject, ConnectionRequestInfo cri)
      throws ResourceException
   {
      Properties props = getConnectionProperties(subject, cri);
      // Some friendly drivers (Oracle, you guessed right) modify the props you supply.
      // Since we use our copy to identify compatibility in matchManagedConnection, we need
      // a pristine copy for our own use.  So give the friendly driver a copy.
      Properties copy = (Properties)props.clone();
      if(log.isDebugEnabled())
      {
         // Make yet another copy to mask the password
         Properties logCopy = copy;
         if(copy.getProperty("password") != null)
         {
            logCopy = (Properties)props.clone();
            logCopy.setProperty("password", "--hidden--");
         }
         log.debug("Using properties: " + logCopy);
      }

      return doCreateManagedConnection(copy, props);
   }

   private ManagedConnection doCreateManagedConnection(Properties copy, Properties props)
      throws JBossResourceException
   {
      // try to get a connection as many times as many urls we have in the list
      for(int i = 0; i < urlSelector.getUrlList().size(); ++i)
      {
         String url = urlSelector.getUrl();

         if(log.isTraceEnabled())
         {
            log.trace("Trying to create a connection to " + url);
         }

         try
         {
            Driver d = getDriver(url);
            Connection con = d.connect(url, copy);
            if(con == null)
            {
               log.warn("Wrong driver class for this connection URL: " + url);
               urlSelector.failedUrl(url);
            }
            else
            {
               return new LocalManagedConnection(this, con, props, transactionIsolation, preparedStatementCacheSize);
            }
         }
         catch(Exception e)
         {
            log.warn("Failed to create connection for " + url + ": " + e.getMessage());
            urlSelector.failedUrl(url);
         }
      }

      // we have supposedly tried all the urls
      throw new JBossResourceException(
         "Could not create connection using any of the URLs: " + urlSelector.getUrlList()
      );
   }

   private void initUrlSelector()
   {
      List urlsList = new ArrayList();
      String urlsStr = getConnectionURL();
      String url;
      int urlStart = 0;
      int urlEnd = urlsStr.indexOf(urlDelimeter);
      while(urlEnd > 0)
      {
         url = urlsStr.substring(urlStart, urlEnd);
         urlsList.add(url);
         urlStart = ++urlEnd;
         urlEnd = urlsStr.indexOf(urlDelimeter, urlEnd);
         log.debug("added HA connection url: " + url);
      }

      if(urlStart != urlsStr.length())
      {
         url = urlsStr.substring(urlStart, urlsStr.length());
         urlsList.add(url);
         log.debug("added HA connection url: " + url);
      }

      this.urlSelector = new URLSelector(urlsList);
   }

   // Inner

   public static class URLSelector
   {
      private final List urls;
      private int urlIndex;
      private String url;

      public URLSelector(List urls)
      {
         if(urls == null || urls.size() == 0)
         {
            throw new IllegalStateException("Expected non-empty list of connection URLs but got: " + urls);
         }
         this.urls = Collections.unmodifiableList(urls);
      }

      public synchronized String getUrl()
      {
         if(url == null)
         {
            if(urlIndex == urls.size())
            {
               urlIndex = 0;
            }
            url = (String)urls.get(urlIndex++);
         }
         return url;
      }

      public synchronized void failedUrl(String url)
      {
         if(url.equals(this.url))
         {
            this.url = null;
         }
      }

      public List getUrlList()
      {
         return urls;
      }
   }
}
