/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webservice.deployment;

// $Id$

import org.jboss.axis.Constants;
import org.jboss.logging.Logger;
import org.jboss.xb.binding.NamespaceRegistry;

import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.io.PrintWriter;

/**
 * Abstracts an Axis service description
 *
 * @author thomas.diesler@jboss.org
 * @since 08-June-2004
 */
public class OperationDescription
{
   // provide logging
   private final Logger log = Logger.getLogger(OperationDescription.class);

   // The java name of the operation
   private String name;
   private QName qname;
   private QName returnQName;
   private QName returnType;
   private boolean oneWay;

   private ArrayList parameters = new ArrayList();
   private ArrayList faults = new ArrayList();

   public OperationDescription(String name, QName qname)
   {
      if (name == null)
         throw new IllegalArgumentException("Operation name cannot be null");

      if (qname == null)
         qname = new QName(name);

      this.name = name;
      this.qname = qname;
   }

   public String getJavaName()
   {
      return name;
   }

   public String getWsdlName()
   {
      return getQName().getLocalPart();
   }

   public QName getQName()
   {
      return qname;
   }

   public QName getReturnQName()
   {
      return returnQName;
   }

   public QName getReturnType()
   {
      return returnType;
   }

   public void setReturnQName(QName returnQName)
   {
      this.returnQName = returnQName;
   }

   public void setReturnType(QName returnType)
   {
      this.returnType = returnType;
   }

   public boolean isOneWay()
   {
      return oneWay;
   }

   public void setOneWay(boolean oneWay)
   {
      this.oneWay = oneWay;
   }

   public Iterator getParameters()
   {
      return parameters.iterator();
   }

   public void setParameters(List newList)
   {
      List oldList = parameters;
      parameters = new ArrayList();

      Iterator itNew = newList.iterator();
      while (itNew.hasNext())
      {
         Parameter param = (Parameter)itNew.next();
         log.debug("Reordering: " + param.getName());
         parameters.add(param);
         oldList.remove(param);
      }

      Iterator itOld = oldList.iterator();
      while (itOld.hasNext())
      {
         Parameter param = (Parameter)itOld.next();
         if (param.inHeader || param.outHeader)
         {
            log.debug("Appending header: " + param.getName());
            parameters.add(param);
         }
         else
         {
            throw new IllegalStateException("Incomplete reorder list: " + param.getName());
         }
      }
   }

   public void addParameter(Parameter param)
   {
      parameters.add(param);
   }

   public void addFault(Fault fault)
   {
      faults.add(fault);
   }

   public Parameter getParameterForName(String partName)
   {
      Parameter param = null;
      Iterator it = parameters.iterator();
      while (param == null && it.hasNext())
      {
         Parameter pa = (Parameter)it.next();
         if (pa.getName().equals(partName))
            param = pa;
      }
      return param;
   }

   public Iterator getFaults()
   {
      return faults.iterator();
   }

   public void writeWSDD(PrintWriter out)
   {
      String name = getJavaName();
      QName opQName = getQName();
      QName returnQName = getReturnQName();
      QName returnType = getReturnType();

      log.trace("Operation: " + this);

      NamespaceRegistry nsRegistry = new NamespaceRegistry();
      opQName = nsRegistry.registerQName(new QName(opQName.getNamespaceURI(), opQName.getLocalPart()));

      if (isUserNamespace(returnQName))
      {
         returnQName = new QName(returnQName.getNamespaceURI(), returnQName.getLocalPart());
         returnQName = nsRegistry.registerQName(returnQName);
      }

      if (isUserNamespace(returnType))
      {
         returnType = new QName(returnType.getNamespaceURI(), returnType.getLocalPart());
         returnType = nsRegistry.registerQName(returnType);
      }

      String opNS = "";
      String opPrefix = opQName.getPrefix();
      if (opPrefix.length() > 0)
      {
         opNS = "xmlns:" + opPrefix + "='" + opQName.getNamespaceURI() + "' ";
      }

      String qnameAttr = WSDDGenerator.getQNameAttrValue(opQName);

      if (returnType != null)
      {
         String returnAttr = WSDDGenerator.getQNameAttrValue(returnQName);
         String typeAttr = WSDDGenerator.getQNameAttrValue(returnType);

         if (isUserNamespace(returnType))
         {
            String typeNS = "xmlns:" + returnType.getPrefix() + "='" + returnType.getNamespaceURI() + "' ";
            if (opNS.indexOf(typeNS) < 0)
            {
               opNS += typeNS;
            }
         }

         out.println("  <operation name='" + name + "' qname='" + qnameAttr + "' returnQName='" + returnAttr + "' returnType='" + typeAttr + "' " + opNS + ">");
      }
      else
      {
         String onewayStr = (oneWay ? "oneway='true' " : "");
         out.println("  <operation name='" + name + "' qname='" + qnameAttr + "' " + onewayStr + opNS + ">");
      }

      Iterator itParam = getParameters();
      while (itParam.hasNext())
      {
         OperationDescription.Parameter param = (OperationDescription.Parameter)itParam.next();
         String paramName = param.getName();
         QName paramQName = param.getQName();
         QName typeQName = param.getType();
         String mode = param.getMode();
         boolean inHeader = param.isInHeader();
         boolean outHeader = param.isOutHeader();

         log.trace("Parameter: " + param);

         if (isUserNamespace(paramQName))
         {
            paramQName = new QName(paramQName.getNamespaceURI(), paramQName.getLocalPart());
            paramQName = nsRegistry.registerQName(paramQName);
         }

         String headers = "";
         if (inHeader)
            headers += "inHeader='true' ";
         if (outHeader)
            headers += "outHeader='true' ";

         String paramNS = "";
         if (typeQName != null)
         {
            if (isUserNamespace(typeQName))
            {
               typeQName = new QName(typeQName.getNamespaceURI(), typeQName.getLocalPart());
               typeQName = nsRegistry.registerQName(typeQName);

               String typeNS = "xmlns:" + typeQName.getPrefix() + "='" + typeQName.getNamespaceURI() + "' ";
               if (opNS.indexOf(typeNS) < 0)
               {
                  paramNS = typeNS;
               }
            }

            String typeAttr = WSDDGenerator.getQNameAttrValue(typeQName);

            qnameAttr = WSDDGenerator.getQNameAttrValue(paramQName);
            out.println("    <parameter name='" + paramName + "' qname='" + qnameAttr + "' mode='" + mode + "' type='" + typeAttr + "' " + headers + paramNS + "/>");
         }
         else
         {
            out.println("    <parameter name='" + paramName + "' mode='" + mode + "' " + headers + paramNS + "/>");
         }
      }

      Iterator itFault = getFaults();
      while (itFault.hasNext())
      {
         OperationDescription.Fault fault = (OperationDescription.Fault)itFault.next();
         String faultName = fault.getName();
         String javaType = fault.getJavaType();

         log.trace("Fault: " + fault);

         QName typeQName = fault.getTypeQName();
         QName faultQName = fault.getQName();

         String faultNS = "";
         if (isUserNamespace(typeQName))
         {
            typeQName = new QName(typeQName.getNamespaceURI(), typeQName.getLocalPart());
            typeQName = nsRegistry.registerQName(typeQName);

            String typeNS = "xmlns:" + typeQName.getPrefix() + "='" + typeQName.getNamespaceURI() + "' ";
            if (opNS.indexOf("xmlns:" + typeNS) < 0)
            {
               faultNS = typeNS;
            }
         }

         String typeAttr = WSDDGenerator.getQNameAttrValue(typeQName);

         if (isUserNamespace(faultQName))
         {
            faultQName = new QName(faultQName.getNamespaceURI(), faultQName.getLocalPart());
            faultQName = nsRegistry.registerQName(faultQName);

            String typeNS = "xmlns:" + faultQName.getPrefix() + "='" + faultQName.getNamespaceURI() + "' ";
            if (opNS.indexOf("xmlns:" + typeNS) < 0)
            {
               faultNS = typeNS;
            }
         }

         qnameAttr = WSDDGenerator.getQNameAttrValue(faultQName);
         out.println("    <fault name='" + faultName + "' qname='" + qnameAttr + "' type='" + typeAttr + "' class='" + javaType + "' " + faultNS + "/>");
      }

      out.println("  </operation>");
   }

   private boolean isUserNamespace(QName qname)
   {
      if (qname == null)
         return false;

      String nsURI = qname.getNamespaceURI();
      boolean isRegisteredURI = "".equals(nsURI);
      isRegisteredURI = isRegisteredURI || Constants.URI_DEFAULT_SCHEMA_XSD.equals(nsURI);
      isRegisteredURI = isRegisteredURI || Constants.URI_SOAP11_ENC.equals(nsURI);
      return isRegisteredURI == false;
   }

   public boolean equals(Object obj)
   {
      if (obj == null)
         return false;
      return toString().equals(obj.toString());
   }

   public int hashCode()
   {
      return toString().hashCode();
   }

   public String toString()
   {
      return "[name=" + name + ",qname=" + qname + ",returnQName=" + returnQName + ",returnType=" + returnType + "]";
   }

   // Used in wsdd operations
   public static class Parameter
   {
      private String name;
      private QName qname;
      private String mode;
      private QName type;
      private boolean inHeader;
      private boolean outHeader;

      public Parameter(String name, QName qname, String mode, QName type)
      {
         if (name == null)
            throw new IllegalArgumentException("Parameter name cannot be null");
         if (mode == null)
            throw new IllegalArgumentException("Parameter mode cannot be null");
         if (type == null)
            throw new IllegalArgumentException("Parameter type cannot be null");

         if (qname == null)
            qname = new QName(name);

         this.name = name;
         this.qname = qname;
         this.mode = mode;
         this.type = type;
      }

      public String getName()
      {
         return name;
      }

      public QName getQName()
      {
         return qname;
      }

      public QName getType()
      {
         return type;
      }

      public String getMode()
      {
         return mode;
      }

      public void setMode(String mode)
      {
         this.mode = mode;
      }

      public boolean isInHeader()
      {
         return inHeader;
      }

      public void setInHeader(boolean inHeader)
      {
         this.inHeader = inHeader;
      }

      public boolean isOutHeader()
      {
         return outHeader;
      }

      public void setOutHeader(boolean outHeader)
      {
         this.outHeader = outHeader;
      }

      public boolean equals(Object obj)
      {
         if (obj == null)
            return false;
         return toString().equals(obj.toString());
      }

      public int hashCode()
      {
         return toString().hashCode();
      }

      public String toString()
      {
         return "[name=" + name + "qname=" + qname + ",mode=" + mode + ",type=" + type + "]";
      }
   }

   // Used in wsdd operations
   public static class Fault
   {
      private String name;
      private QName qname;
      private QName typeQName;
      private String javaType;

      public Fault(String name, QName qname, String javaType, QName type)
      {
         if (name == null)
            throw new IllegalArgumentException("Fault name cannot be null");
         if (javaType == null)
            throw new IllegalArgumentException("Fault javaType cannot be null");
         if (type == null)
            throw new IllegalArgumentException("Fault type cannot be null");

         if (qname == null)
            qname = new QName(name);

         this.name = name;
         this.qname = qname;
         this.javaType = javaType;
         this.typeQName = type;
      }

      public String getName()
      {
         return name;
      }

      public QName getQName()
      {
         return qname;
      }

      public QName getTypeQName()
      {
         return typeQName;
      }

      public String getJavaType()
      {
         return javaType;
      }

      public boolean equals(Object obj)
      {
         if (obj == null)
            return false;
         return toString().equals(obj.toString());
      }

      public int hashCode()
      {
         return toString().hashCode();
      }

      public String toString()
      {
         return "[name=" + name + "qname=" + qname + ",type=" + typeQName + ",java=" + javaType + "]";
      }
   }
}