/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.webservice.deployment;

// $Id$

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.jboss.axis.utils.LinkedHashMap;
import org.jboss.util.xml.DOMUtils;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Additional bean serialization meta data.
 *
 * Here is an example:
 *
 * <typeMapping
 *    qname='ns1:SequenceStruct2' xmlns:ns1='http://MarshallTest.org/xsd'
 *    type='java:org.jboss.ws4ee.test.marshalltest.rpcenc.SequenceStruct2'
 *    serializer='org.jboss.webservice.encoding.ser.MetaDataBeanSerializerFactory'
 *    deserializer='org.jboss.webservice.encoding.ser.MetaDataBeanDeserializerFactory'
 *    encodingStyle='http://schemas.xmlsoap.org/soap/encoding/'>
 *    <typeDesc>
 *       <elementDesc fieldName="varBase64Binary" xmlName="varBase64Binary" xmlType="xsd:base64Binary"/>
 *       <elementDesc fieldName="varHexBinary" xmlName="varHexBinary" xmlType="xsd:hexBinary"/>
 *       <elementDesc fieldName="varSoapString" xmlName="varSoapString" xmlType="soapenc:string"/>
 *       <elementDesc fieldName="varSoapBoolean" xmlName="varSoapBoolean" xmlType="soapenc:boolean"/>
 *       <elementDesc fieldName="varSoapFloat" xmlName="varSoapFloat" xmlType="soapenc:float"/>
 *       <elementDesc fieldName="varSoapDouble" xmlName="varSoapDouble" xmlType="soapenc:double"/>
 *       <elementDesc fieldName="varSoapDecimal" xmlName="varSoapDecimal" xmlType="soapenc:decimal"/>
 *       <elementDesc fieldName="varSoapInt" xmlName="varSoapInt" xmlType="soapenc:int"/>
 *       <elementDesc fieldName="varSoapShort" xmlName="varSoapShort" xmlType="soapenc:short"/>
 *       <elementDesc fieldName="varSoapByte" xmlName="varSoapByte" xmlType="soapenc:byte"/>
 *       <elementDesc fieldName="varSoapBase64" xmlName="varSoapBase64" xmlType="soapenc:base64"/>
 *       <elementDesc fieldName="varDateTimeArray" xmlName="varDateTimeArray" itemXmlType="xsd:datetime"/>
 *       <elementOrder>
 *          <element name="varString"/>
 *          <element name="varInteger"/>
 *          <element name="varInt"/>
 *          <element name="varLong"/>
 *          <element name="varShort"/>
 *          <element name="varDecimal"/>
 *          <element name="varFloat"/>
 *          <element name="varDouble"/>
 *          <element name="varBoolean"/>
 *          <element name="varByte"/>
 *          <element name="varQName"/>
 *          <element name="varDateTime"/>
 *          <element name="varSoapString"/>
 *          <element name="varSoapBoolean"/>
 *          <element name="varSoapFloat"/>
 *          <element name="varSoapDouble"/>
 *          <element name="varSoapDecimal"/>
 *          <element name="varSoapInt"/>
 *          <element name="varSoapShort"/>
 *          <element name="varSoapByte"/>
 *          <element name="varBase64Binary"/>
 *          <element name="varHexBinary"/>
 *          <element name="varSoapBase64"/>
 *          <element name="varSequenceStruct"/>
 *          <element name="varDateTimeArray"/>
 *       </elementOrder>
 *    </typeDesc>
 * </typeMapping>
 *
 * @author thomas.diesler@jboss.org
 * @since 18-June-2004
 */
public class BeanXMLMetaData
{
   private Map elementDescMap = new LinkedHashMap();

   // List<String> of field names
   private List elementOrderList = new ArrayList();

   // Hide constructoer
   private BeanXMLMetaData()
   {
   }

   /**
    * Parse the bean meta data from XML
    */
   public static BeanXMLMetaData parse(Element metaData)
   {
      BeanXMLMetaData beanMetaData = new BeanXMLMetaData();

      if (metaData != null)
      {
         ArrayList qnames = new ArrayList();

         // Parse element descriptions
         NodeList nlistElementDesc = metaData.getElementsByTagName("elementDesc");
         for (int i = 0; i < nlistElementDesc.getLength(); i++)
         {
            Element elDesc = (Element)nlistElementDesc.item(i);
            String fieldName = DOMUtils.getAttributeValue(elDesc, "fieldName");

            QName xmlName = DOMUtils.getAttributeValueAsQName(elDesc, "xmlName");
            if (xmlName == null)
               xmlName = new QName("", fieldName);

            QName xmlType = DOMUtils.getAttributeValueAsQName(elDesc, "xmlType");
            if (xmlType != null)
            {
               if (qnames.contains(xmlType))
                  throw new IllegalStateException("Duplicate type mapping definition: " + xmlType);

               qnames.add(xmlType);
            }

            ElementMetaData elMetaData = new ElementMetaData(fieldName, xmlName, xmlType);
            beanMetaData.elementDescMap.put(fieldName, elMetaData);

            boolean asAttribute = DOMUtils.getAttributeValueAsBoolean(elDesc, "asAttr");
            elMetaData.setAsAttribute(asAttribute);

            boolean asContent = DOMUtils.getAttributeValueAsBoolean(elDesc, "asContent");
            elMetaData.setAsContent(asContent);

            Integer minOccurs = DOMUtils.getAttributeValueAsInteger(elDesc, "minOccurs");
            elMetaData.setMinOccurs(minOccurs);

            QName itemXmlType = DOMUtils.getAttributeValueAsQName(elDesc, "itemXmlType");
            elMetaData.setItemXmlType(itemXmlType);
         }

         // Parse element order
         Element orderEl = DOMUtils.getFirstChildElement(metaData, "elementOrder");
         if (orderEl != null)
         {
            // copy the BeanPropertyDescriptors according to the meta data
            // elementOrder
            NodeList nlistElementOrder = orderEl.getElementsByTagName("element");
            for (int i = 0; i < nlistElementOrder.getLength(); i++)
            {
               Element el = (Element)nlistElementOrder.item(i);
               String fieldName = el.getAttribute("name");
               beanMetaData.elementOrderList.add(fieldName);
            }
         }
      }

      return beanMetaData;
   }

   public Iterator getElementMetaData()
   {
      return elementDescMap.values().iterator();
   }

   public ElementMetaData getElementMetaDataByFieldName(String fieldName)
   {
      ElementMetaData elMetaData = null;
      Iterator it = elementDescMap.keySet().iterator();
      while (it.hasNext())
      {
         String key = (String)it.next();
         if (key.equalsIgnoreCase(fieldName))
         {
            elMetaData = (ElementMetaData)elementDescMap.get(key);
            break;
         }
      }
      return elMetaData;
   }

   public List getElementOrder()
   {
      return new ArrayList(elementOrderList);
   }

   public void serializeAsXML(PrintWriter out)
   {
      String pad = "  ";
      out.println(pad + "<typeDesc>");
      Iterator itElDesc = elementDescMap.values().iterator();
      while (itElDesc.hasNext())
      {
         ElementMetaData elMetaData = (ElementMetaData)itElDesc.next();
         QName xmlName = elMetaData.getXmlName();

         String nsDecl = "";
         String prefix = xmlName.getPrefix();
         if (prefix.length() > 0)
         {
            nsDecl = " xmlns:" + prefix + "='" + xmlName.getNamespaceURI() + "'";
         }
         
         String nameAttr = WSDDGenerator.getQNameAttrValue(xmlName);
         out.print(pad + pad + "<elementDesc fieldName='" + elMetaData.getFieldName() + "' xmlName='" + nameAttr + "'");

         QName xmlType = elMetaData.getXmlType();
         if (xmlType != null)
         {
            String typeAttr = WSDDGenerator.getQNameAttrValue(xmlType);
            out.print(" xmlType='" + typeAttr + "'");
         }

         if (elMetaData.isAsAttribute())
            out.print(" asAttr='true'");

         if (elMetaData.isAsContent())
            out.print(" asContent='true'");

         if (elMetaData.getMinOccurs() != null)
            out.print(" minOccurs='" + elMetaData.getMinOccurs() + "'");

         if (elMetaData.getItemXmlType() != null)
         {
            out.print(" itemXmlType='" + elMetaData.getPrefixedItemXmlType() + "'");
         }

         out.println(nsDecl + "/>");
      }

      if (elementOrderList.size() > 0)
      {
         out.println(pad + pad + "<elementOrder>");
         Iterator itElOrder = elementOrderList.iterator();
         while (itElOrder.hasNext())
         {
            String fieldName = (String)itElOrder.next();
            out.println(pad + pad + pad + "<element name='" + fieldName + "'/>");
         }
         out.println(pad + pad + "</elementOrder>");
      }
      out.println(pad + "</typeDesc>");
   }

   /**
    * An XML element description
    */
   public static class ElementMetaData
   {

      private String fieldName;

      private QName xmlName;

      private QName xmlType;

      private QName itemXmlType;

      private boolean asAttribute;

      private boolean asContent;

      private Integer minOccurs;

      public ElementMetaData(String fieldName, QName xmlName, QName xmlType)
      {
         this.fieldName = fieldName;
         this.xmlName = xmlName;
         this.xmlType = xmlType;
      }

      public String getFieldName()
      {
         return fieldName;
      }

      public QName getXmlName()
      {
         return xmlName;
      }

      public String getPrefixedXmlName()
      {
         String retStr = null;
         if (xmlName != null)
         {
            if (xmlName.getPrefix().length() > 0)
               retStr = xmlName.getPrefix() + ":" + xmlName.getLocalPart();
            else
               retStr = xmlName.getLocalPart();
         }
         return retStr;
      }

      public QName getXmlType()
      {
         return xmlType;
      }

      public boolean isAsAttribute()
      {
         return asAttribute;
      }

      public void setAsAttribute(boolean asAttribute)
      {
         this.asAttribute = asAttribute;
      }

      public boolean isAsContent()
      {
         return asContent;
      }

      public void setAsContent(boolean asContent)
      {
         this.asContent = asContent;
      }

      public Integer getMinOccurs()
      {
         return minOccurs;
      }

      public void setMinOccurs(Integer minOccurs)
      {
         this.minOccurs = minOccurs;
      }

      public QName getItemXmlType()
      {
         return itemXmlType;
      }

      public String getPrefixedItemXmlType()
      {
         String retStr = null;
         if (itemXmlType != null)
         {
            retStr = itemXmlType.getLocalPart();
            if (itemXmlType.getPrefix().length() > 0)
               retStr = itemXmlType.getPrefix() + ":" + retStr;
         }
         return retStr;
      }

      /**
       * Solves mapping problems were a nested type in an array can not be properly determined
       */
      public void setItemXmlType(QName itemXmlType)
      {
         this.itemXmlType = itemXmlType;
      }
   }
}
