/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webservice;

// $Id$

import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.InstanceNotFoundException;
import javax.management.Notification;
import javax.management.NotificationFilterSupport;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import org.dom4j.Document;
import org.dom4j.Element;
import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.MainDeployerMBean;
import org.jboss.deployment.SubDeployer;
import org.jboss.logging.Logger;
import org.jboss.metadata.ApplicationMetaData;
import org.jboss.metadata.BeanMetaData;
import org.jboss.metadata.EjbPortComponentMetaData;
import org.jboss.metadata.WebMetaData;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.webservice.metadata.PortComponentMetaData;
import org.jboss.webservice.metadata.WebserviceDescriptionMetaData;
import org.jboss.webservice.metadata.WebservicesFactory;
import org.jboss.webservice.metadata.WebservicesMetaData;
import org.jboss.xb.binding.ObjectModelFactory;
import org.jboss.xb.binding.Unmarshaller;
import org.jboss.xb.binding.UnmarshallerFactory;

/**
 * A deployer service that manages WS4EE compliant Web-Services within JMX
 * by translating/delegating to an axis deployer.
 * <p/>
 * This service receives deployment notifications from the EJBDeployer and
 * AbstractWebContainer and deploys the webservices using the {@link AxisService}
 *
 * @author Thomas.Diesler@jboss.org
 * @author Scott.Stark@jboss.org
 * @since 15-April-2004
 */
public abstract class ServiceDeployer extends ServiceMBeanSupport implements ServiceDeployerMBean, NotificationListener
{
   // provide logging
   private final Logger log = Logger.getLogger(ServiceDeployer.class);

   // The servlet init param in web.xml that is the service ID
   public static final String INIT_PARAM_SERVICE_ENDPOINT_ID = "ServiceEndpointID";
   // The servlet init param in web.xml that is the service endpoint class
   public static final String INIT_PARAM_SERVICE_ENDPOINT_IMPL = "ServiceEndpointImpl";

   // Proxy to AxisService
   private AxisServiceMBean axisService;

   /**
    * Maps the deployment url the the WebservicesMetaData
    */
   protected Map webservicesMap = new HashMap();

   /** Get a proxy to AxisService
    */
   protected void startService() throws Exception
   {
      super.startService();
      axisService = (AxisServiceMBean)MBeanProxyExt.create(AxisServiceMBean.class, AxisServiceMBean.OBJECT_NAME, server);
   }

   /**
    * Callback method from the broadcaster MBean this listener implementation
    * is registered to.
    *
    * @param notification the notification object
    * @param handback     the handback object given to the broadcaster
    *                     upon listener registration
    */
   public void handleNotification(Notification notification, Object handback)
   {
      DeploymentInfo di = (DeploymentInfo)notification.getUserData();

      String moduleName = di.shortName;
      String type = notification.getType();
      log.debug("handleNotification: " + type + "," + moduleName);

      if (isWebservicesDeployment(di))
      {
         try
         {
            if (type.equals(SubDeployer.CREATE_NOTIFICATION))
               createWebservice(di);
            else if (type.equals(SubDeployer.START_NOTIFICATION))
               startWebservice(di);
         }
         catch (Throwable e)
         {
            handleStartupException(di, e);
         }

         try
         {
            if (type.equals(SubDeployer.STOP_NOTIFICATION))
               stopWebservice(di);
            else if (type.equals(SubDeployer.DESTROY_NOTIFICATION))
               destroyWebservice(di);
         }
         catch (Throwable e)
         {
            handleShutdownException(moduleName, e);
         }
      }
   }

   /**
    * Overwrite to create the webservice
    * Is called when the parent deployer sends the CREATE_NOTIFICATION.
    * <p/>
    * This implementation parses webservices.xml and puts it in the local registry.
    */
   protected void createWebservice(DeploymentInfo di) throws DeploymentException
   {
      URL url = getWebservicesDescriptor(di);
      if (url != null)
      {
         WebservicesMetaData wsMetaData = parseWebservicesXML(di, url);
         webservicesMap.put(di.url, wsMetaData);
      }
   }

   /** Get the resource name of the webservices.xml descriptor. */
   protected abstract URL getWebservicesDescriptor(DeploymentInfo di);

   /** Return true if this is a web service deployment */
   private boolean isWebservicesDeployment(DeploymentInfo di)
   {
      boolean isWebservicesDeployment = false;
      if (di.metaData instanceof ApplicationMetaData)
      {
         ApplicationMetaData applMetaData = (ApplicationMetaData)di.metaData;
         isWebservicesDeployment = applMetaData.isWebServiceDeployment() || getWebservicesDescriptor(di) != null;
      }

      if (di.metaData instanceof WebMetaData)
      {
         WebMetaData webMetaData = (WebMetaData)di.metaData;
         isWebservicesDeployment = webMetaData.isWebServiceDeployment() || getWebservicesDescriptor(di) != null;
      }

      return isWebservicesDeployment;
   }

   /**
    * Overwrite to start the webservice
    * Is called when the parent deployer sends the START_NOTIFICATION.
    * <p/>
    * This implementation deployes the webservices to Axis.
    */
   protected void startWebservice(DeploymentInfo di) throws DeploymentException
   {
      WebservicesMetaData webservices = (WebservicesMetaData)webservicesMap.get(di.url);
      if (webservices != null)
      {
         deployWebservices(di, webservices);

         // update the service address
         ServiceLocationResolver locationResolver = new ServiceLocationResolver(di);
         WebserviceDescriptionMetaData[] wsdArray = webservices.getWebserviceDescriptions();
         for (int i = 0; i < wsdArray.length; i++)
         {
            WebserviceDescriptionMetaData wsdMetaData = wsdArray[i];
            wsdMetaData.updateServiceAddress(locationResolver);
            String wsdName = wsdMetaData.getWebserviceDescriptionName();

            // copy the wsdl publish location from jboss.xml
            if (di.metaData instanceof ApplicationMetaData)
            {
               ApplicationMetaData applMetaData = (ApplicationMetaData)di.metaData;
               String wsdlPublishLocation = applMetaData.getWsdlPublishLocationByName(wsdName);
               wsdMetaData.setWsdlPublishLocation(wsdlPublishLocation);
            }

            // copy the wsdl publish location from jboss-web.xml
            if (di.metaData instanceof WebMetaData)
            {
               WebMetaData webMetaData = (WebMetaData)di.metaData;
               String wsdlPublishLocation = webMetaData.getWsdlPublishLocationByName(wsdName);
               wsdMetaData.setWsdlPublishLocation(wsdlPublishLocation);
            }
         }

         WSDLFilePublisher wsdlfp = new WSDLFilePublisher(di);
         wsdlfp.publishWsdlFile(webservices);

         // Log the deployed services
         for (int i = 0; i < wsdArray.length; i++)
         {
            WebserviceDescriptionMetaData wsdMetaData = wsdArray[i];
            PortComponentMetaData[] pcArr = wsdMetaData.getPortComponents();
            for (int j = 0; j < pcArr.length; j++)
            {
               PortComponentMetaData pcMetaData = pcArr[j];
               URL endpointURL = pcMetaData.getServiceEndpointURL();
               log.info("Web Service deployed: " + endpointURL);
            }
         }
      }
   }

   /**
    * Overwrite to stop the webservice
    * Is called when the parent deployer sends the STOP_NOTIFICATION.
    * <p/>
    * This implementation undeployes the webservices to Axis and
    * removes the webservices.xml from the local registry.
    */
   protected void stopWebservice(DeploymentInfo di)
   {
      WebservicesMetaData webservices = (WebservicesMetaData)webservicesMap.get(di.url);
      if (webservices != null)
      {
         undeployWebservices(di, webservices);
         WSDLFilePublisher wsdlfp = new WSDLFilePublisher(di);
         wsdlfp.unpublishWsdlFile();
      }
   }

   /**
    * Overwrite to destroy the webservice
    * This method is called when the parent deployer sends the DESTROY_NOTIFICATION.
    */
   protected void destroyWebservice(DeploymentInfo di)
   {
      webservicesMap.remove(di.url);
   }

   // PROTECTED  *******************************************************************************************************

   /**
    * Handle all webservice deployment exceptions.
    * <p/>
    * You can either simply logs the problem and keep the EJB/WAR module
    * alive or undeploy properly.
    */
   protected void handleStartupException(DeploymentInfo di, Throwable th)
   {
      log.error("Cannot startup webservice for: " + di.shortName, th);
      try
      {
         MainDeployerMBean mainDeployer = (MainDeployerMBean)MBeanProxyExt.create(MainDeployerMBean.class, MainDeployerMBean.OBJECT_NAME, server);
         mainDeployer.undeploy(di);
      }
      catch (Exception e)
      {
         log.warn("Error handling exception", e);
      }
   }

   /**
    * Handle all webservice deployment exceptions.
    * <p/>
    * You can either simply logs the problem and keep the EJB/WAR module
    * alive or undeploy properly.
    */
   protected void handleShutdownException(String moduleName, Throwable th)
   {
      log.error("Cannot shutdown webservice for: " + moduleName, th);
   }

   /**
    * Register the notification listener
    */
   protected void registerNotificationListener(ObjectName serviceName) throws InstanceNotFoundException
   {
      NotificationFilterSupport filter = new NotificationFilterSupport();
      filter.enableType(SubDeployer.INIT_NOTIFICATION);
      filter.enableType(SubDeployer.CREATE_NOTIFICATION);
      filter.enableType(SubDeployer.START_NOTIFICATION);
      filter.enableType(SubDeployer.STOP_NOTIFICATION);
      filter.enableType(SubDeployer.DESTROY_NOTIFICATION);
      server.addNotificationListener(serviceName, this, filter, null);
   }

   /**
    * Unregister the notification listener
    */
   protected void unregisterNotificationListener(ObjectName serviceName)
   {
      try
      {
         server.removeNotificationListener(serviceName, this);
      }
      catch (Exception e)
      {
         log.error("Cannot remove notification listener: " + e.toString());
      }
   }

   /**
    * Unmarshal the webservices.xml
    */
   protected WebservicesMetaData parseWebservicesXML(DeploymentInfo di, URL webservicesURL) throws DeploymentException
   {
      WebservicesMetaData webservices = null;
      try
      {
         // let the object model factory to create an instance and populate it with data from XML
         InputStream is = webservicesURL.openStream();
         try
         {
            // setup the XML binding Unmarshaller
            Unmarshaller unmarshaller = UnmarshallerFactory.newInstance().newUnmarshaller();
            ObjectModelFactory factory = new WebservicesFactory(di.localCl);
            webservices = (WebservicesMetaData)unmarshaller.unmarshal(is, factory, null);
         }
         finally
         {
            is.close();
         }
      }
      catch (Exception e)
      {
         throw new DeploymentException("Cannot obtain webservices meta data", e);
      }
      return webservices;
   }

   /**
    * Deploy the webservices using the AxisService MBean
    */
   protected void deployWebservices(DeploymentInfo di, WebservicesMetaData webservices) throws DeploymentException
   {
      try
      {
         WebserviceDescriptionMetaData[] wsdArr = webservices.getWebserviceDescriptions();
         for (int i = 0; i < wsdArr.length; i++)
         {
            WebserviceDescriptionMetaData wsd = wsdArr[i];
            PortComponentMetaData[] pcArr = wsd.getPortComponents();
            for (int j = 0; j < pcArr.length; j++)
            {
               PortComponentMetaData pcMetaData = pcArr[j];
               PortComponentInfo pcInfo = new PortComponentInfo(di, pcMetaData);
               axisService.deployService(pcInfo);
            }
         }
      }
      catch (Exception e)
      {
         throw new DeploymentException("Cannot deploy webservice", e);
      }
   }

   /**
    * Undeploy the webservices using the AxisService MBean
    */
   protected void undeployWebservices(DeploymentInfo di, WebservicesMetaData webservices)
   {
      try
      {
         WebserviceDescriptionMetaData[] wsdarr = webservices.getWebserviceDescriptions();
         for (int i = 0; i < wsdarr.length; i++)
         {
            WebserviceDescriptionMetaData wsDescription = wsdarr[i];
            PortComponentMetaData[] pcarr = wsDescription.getPortComponents();
            for (int j = 0; j < pcarr.length; j++)
            {
               PortComponentMetaData pcMetaData = pcarr[j];
               PortComponentInfo pcInfo = new PortComponentInfo(di, pcMetaData);
               String wsID = pcInfo.getServiceID();
               axisService.undeployService(wsID);
            }
         }
      }
      catch (Exception ignore)
      {
         log.warn("Cannot undeploy webservice: " + ignore);
      }
   }

   /** Modify the servlet-class element
    */
   protected boolean modifyServletConfig(Document doc, String servletName, PortComponentInfo pcInfo) throws DeploymentException
   {
      Element servletElement = null;

      Iterator itServlet = doc.getRootElement().elements("servlet").iterator();
      while (itServlet.hasNext() && servletElement == null)
      {
         Element elServlet = (Element)itServlet.next();
         String elName = elServlet.elementTextTrim("servlet-name");
         if (servletName.equals(elName))
            servletElement = elServlet;
      }
      if (servletElement == null)
         throw new DeploymentException("Cannot find <servlet> with servlet-name: " + servletName);

      // find the servlet-class
      Element classElement = servletElement.element("servlet-class");
      if (classElement == null)
         throw new DeploymentException("Cannot find <servlet-class> for servlet-name: " + servletName);

      // replace the class name
      String servletClass = classElement.getTextTrim();
      String serviceEndpointServletName = getServiceEndpointServletName();

      // Nothing to do if we have an <init-param> with the WebServiceID
      if (isAlreadyModified(servletElement) == false)
      {
         classElement.setText(serviceEndpointServletName);

         // build a list of detached elements that come after <servlet-class>
         boolean startDetach = false;
         ArrayList detachedElements = new ArrayList();
         Iterator itServletChildren = servletElement.elements().iterator();
         while (itServletChildren.hasNext())
         {
            Element elServletChild = (Element)itServletChildren.next();
            if (startDetach == true)
            {
               detachedElements.add(elServletChild);
               elServletChild.detach();
            }
            if (elServletChild.equals(classElement))
               startDetach = true;
         }

         // add additional init params
         Element paramElement = servletElement.addElement("init-param");
         paramElement.addElement("param-name").addText(INIT_PARAM_SERVICE_ENDPOINT_ID);
         paramElement.addElement("param-value").addText(pcInfo.getServiceID());

         // In case of the generated web.xml for EJB endpoints
         // we don't have an endpoint implemenation pojo in <servlet-class>
         if (servletClass.equals(serviceEndpointServletName) == false)
         {
            paramElement = servletElement.addElement("init-param");
            paramElement.addElement("param-name").addText(INIT_PARAM_SERVICE_ENDPOINT_IMPL);
            paramElement.addElement("param-value").addText(servletClass);
         }

         PortComponentMetaData pcMetaData = pcInfo.getPortComponentMetaData();
         pcMetaData.setServiceEndpointBean(servletClass);

         // reattach the elements
         Iterator itDetached = detachedElements.iterator();
         while (itDetached.hasNext())
         {
            Element el = (Element)itDetached.next();
            servletElement.add(el);
         }

         return true;
      }
      else
      {
         Iterator itInitParam = servletElement.elementIterator("init-param");
         while (itInitParam.hasNext())
         {
            Element elInitParam = (Element)itInitParam.next();
            if (INIT_PARAM_SERVICE_ENDPOINT_IMPL.equals(elInitParam.elementText("param-name")))
            {
               String serviceEndpointImpl = elInitParam.elementText("param-value");
               PortComponentMetaData pcMetaData = pcInfo.getPortComponentMetaData();
               pcMetaData.setServiceEndpointBean(serviceEndpointImpl);
            }
         }

         return false;
      }
   }

   protected void initTransportGuarantee(Document doc, String servletName, PortComponentInfo pcInfo) throws DeploymentException
   {
      Element servletMapping = null;
      Iterator itServlet = doc.getRootElement().elements("servlet-mapping").iterator();
      while (itServlet.hasNext() && servletMapping == null)
      {
         Element elServletMapping = (Element)itServlet.next();
         String elName = elServletMapping.elementTextTrim("servlet-name");
         if (servletName.equals(elName))
            servletMapping = elServletMapping;
      }
      if (servletMapping != null)
      {
         // find servlet-mapping/url-pattern
         String urlPattern = servletMapping.elementTextTrim("url-pattern");
         if (urlPattern == null)
            throw new DeploymentException("Cannot find <url-pattern> for servlet-name: " + servletName);

         Iterator itSecConstraint = doc.getRootElement().elements("security-constraint").iterator();
         while (itSecConstraint.hasNext())
         {
            Element elSecurityConstraint = (Element)itSecConstraint.next();
            Iterator itWebResourceCollection = elSecurityConstraint.elementIterator("web-resource-collection");
            while (itWebResourceCollection.hasNext())
            {
               Element elWebResourceCollection = (Element)itWebResourceCollection.next();
               String wrcurlPattern = elWebResourceCollection.elementTextTrim("url-pattern");
               if (urlPattern.equals(wrcurlPattern))
               {
                  Element elUserDataConstraint = elSecurityConstraint.element("user-data-constraint");
                  if (elUserDataConstraint != null)
                  {
                     String transportGuarantee = elUserDataConstraint.elementTextTrim("transport-guarantee");
                     PortComponentMetaData pcMetaData = pcInfo.getPortComponentMetaData();
                     pcMetaData.setTransportGuarantee(transportGuarantee);
                  }
               }
            }
         }
      }
   }

   // Return true if the web.xml is already modified
   private boolean isAlreadyModified(Element servletElement)
   {
      String serviceID = null;

      Iterator it = servletElement.elementIterator("init-param");
      while (serviceID == null && it.hasNext())
      {
         Element elParam = (Element)it.next();
         if (INIT_PARAM_SERVICE_ENDPOINT_ID.equals(elParam.elementText("param-name")))
            serviceID = elParam.elementText("param-value");
      }

      return serviceID != null;
   }

   /**
    * Override to return the name of the service endpoint servlet
    */
   protected abstract String getServiceEndpointServletName();

   /**
    * This guy resolves the service location, when ask to do so
    */
   public class ServiceLocationResolver
   {
      private DeploymentInfo di;

      public ServiceLocationResolver(DeploymentInfo di)
      {
         this.di = di;
      }

      public boolean alwaysResolve()
      {
         return axisService.isAlwaysModifySOAPAddress();
      }

      /**
       * Get the web service address for either an JSE or an EJB endpoint
       * <p/>
       * EJB: [schema][host]:[port]/[port-component-uri|deployment name/port-component-name]
       * JSE: [schema][host]:[port]/[path derived from servlet mapping in web.xml]
       *
       * @param schema The url schema, can be 'http://' or 'https://', or null
       * @param pcmd   The port component meta data
       * @return
       */
      public String getServiceLocation(String schema, PortComponentMetaData pcmd)
      {
         String ejbLink = pcmd.getEjbLink();
         String servletLink = pcmd.getServletLink();
         String serviceName = pcmd.getPortComponentName();

         String servicePath = null;

         // For a web based service endpoint it is derived from the servlet mapping
         if (servletLink != null)
         {
            WebMetaData metaData = (WebMetaData)di.metaData;
            Map servletMappings = metaData.getServletMappings();
            String urlPattern = (String)servletMappings.get(servletLink);
            if (urlPattern == null)
               throw new IllegalStateException("Cannot obtain servlet-mapping for: " + servletLink);

            if (urlPattern.startsWith("/") == false)
               urlPattern = "/" + urlPattern;

            servicePath = metaData.getContextRoot() + urlPattern;
         }
         else if (ejbLink != null)
         {
            ApplicationMetaData amd = (ApplicationMetaData)di.metaData;
            BeanMetaData bmd = amd.getBeanByEjbName(ejbLink);
            if (bmd == null)
               throw new IllegalStateException("Cannot find ejb-name: " + ejbLink);

            // Use the webservice context root if we have one
            String contextRoot = amd.getWebServiceContextRoot();

            // If not, derive the context root from the deployment short name
            if (contextRoot == null)
            {
               String shortName = di.shortName;
               contextRoot = shortName.substring(0, shortName.indexOf('.'));
               contextRoot = "/" + contextRoot;
               amd.setWebServiceContextRoot(contextRoot);
            }

            EjbPortComponentMetaData ejbpcMetaData = bmd.getPortComponent();
            if (ejbpcMetaData != null && ejbpcMetaData.getPortComponentURI() != null)
            {
               servicePath = contextRoot + ejbpcMetaData.getPortComponentURI();
            }
            else
            {
               servicePath = contextRoot + "/" + serviceName;
            }
         }
         else
         {
            throw new IllegalStateException("Cannot find valid <servlet-link> nor <ejb-link> in port component meta data");
         }

         if (servicePath.endsWith("/*"))
            servicePath = servicePath.substring(0, servicePath.indexOf("/*"));

         if (schema == null)
            schema = "http://";

         int port = 0;
         String host = null;
         String serviceEndpointAddress = null;
         try
         {
            host = axisService.getWebServiceHost();
            port = axisService.getWebServicePort();
            if ("https://".equals(schema))
               port = axisService.getWebServiceSecurePort();

            serviceEndpointAddress = new URL(schema + host + ":" + port + servicePath).toExternalForm();
         }
         catch (Exception e)
         {
            log.error("Cannot obtain attribute from AxisService, cause: " + e.toString());
         }

         return serviceEndpointAddress;
      }
   }
}
