/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.spring.factory;

import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.DeploymentException;
import org.jboss.naming.NonSerializableFactory;
import org.jboss.naming.Util;
import org.jboss.logging.Logger;
import org.springframework.core.io.UrlResource;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.FatalBeanException;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.util.Map;
import java.util.TreeMap;
import java.net.URL;
import java.net.URLClassLoader;

/**
 * @author <a href="mailto:ales.justin@genera-lynx.com">Ales Justin</a>
 */
public class BeanFactoryLoaderImpl implements BeanFactoryLoader {

    private static final Logger log = Logger.getLogger(BeanFactoryLoaderImpl.class);

    private Map beanFactoryNamesMap = new TreeMap();

    private URL getDocUrl(DeploymentInfo di) throws DeploymentException {
        URL docURL = di.localUrl;
        if (!di.isXML) {
            URLClassLoader localCL = di.localCl;
            docURL = localCL.findResource("META-INF/jboss-spring.xml");
        }
        // Validate that the descriptor was found
        if (docURL == null) {
            throw new DeploymentException("Failed to find META-INF/jboss-spring.xml");
        }
        return docURL;
    }

    private String getDefaultName(DeploymentInfo di) {
        String shortName = di.shortName;
        int p = shortName.indexOf(".spring");
        if (p > 0) {
            return shortName.substring(0, p);
        }
        p = shortName.indexOf("-spring.xml");
        if (p > 0) {
            return shortName.substring(0, p);
        }
        return null;
    }

    public void create(DeploymentInfo di) throws DeploymentException {
        URL docURL = getDocUrl(di);
        String defaultName = getDefaultName(di);
        NamedXmlBeanFactory beanFactory = null;
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(di.ucl);
            beanFactory = new NamedXmlBeanFactory(defaultName, new UrlResource(docURL));
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
        bind(beanFactory, beanFactory.getName());
        log.info("Bean factory [" + beanFactory.getName() + "] binded to local JNDI.");
        beanFactoryNamesMap.put(docURL.toString(), beanFactory.getName());
    }

    public void start(DeploymentInfo di) throws DeploymentException {
    }

    public void stop(DeploymentInfo di) throws DeploymentException {
        try {
            URL docURL = getDocUrl(di);
            String name = (String) beanFactoryNamesMap.remove(docURL.toString());
            ConfigurableBeanFactory beanFactory = lookup(name);
            beanFactory.destroySingletons();
            unbind(name);
            log.info("Bean factory [" + name + "] unbinded from local JNDI.");
        } catch (Exception e) {
            throw new DeploymentException(e);
        }
    }

    public void destroy(DeploymentInfo di) throws DeploymentException {
    }

    // JNDI stuff

    public static void bind(BeanFactory beanFactory, String name) throws BeansException {
        InitialContext ctx = null;
        try {
            ctx = new InitialContext();
            NonSerializableFactory.rebind(ctx, name, beanFactory);
        } catch (NamingException e) {
            throw new FatalBeanException("Unable to bind BeanFactory into JNDI", e);
        } finally {
            if (ctx != null) {
                try {
                    ctx.close();
                } catch (Throwable ignored) {
                }
            }
        }
    }

    public static void unbind(String name) throws BeansException {
        InitialContext ctx = null;
        try {
            ctx = new InitialContext();
            ctx.unbind(name);
            NonSerializableFactory.unbind(name);
        } catch (NamingException e) {
            throw new FatalBeanException("Unable to unbind BeanFactory from JNDI", e);
        } finally {
            if (ctx != null) {
                try {
                    ctx.close();
                } catch (Throwable ignored) {
                }
            }
        }
    }

    public static ConfigurableBeanFactory lookup(String name) throws Exception {
        ConfigurableBeanFactory beanFactory = (ConfigurableBeanFactory) Util.lookup(name, BeanFactory.class);
        log.debug("Found Spring Bean factory [" + name + "]: " + beanFactory);
        return beanFactory;
    }

}
