/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.axis.description;

// $Id$

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.List;

import javax.xml.namespace.QName;

import org.jboss.axis.enums.Style;
import org.jboss.logging.Logger;

/** A helper class to wrap/unwrap document style request/response structures.
 *
 * @author Thomas.Diesler@jboss.org
 * @since 06-Jun-2005
 */
public class ParameterWrapping
{
   // provide logging
   private static Logger log = Logger.getLogger(ParameterWrapping.class);

   public static Object wrapRequestParameters(OperationDesc opDesc, Object[] inParams)
   {
      assertOperationDesc(opDesc);

      ParameterDesc paramDesc = (ParameterDesc)opDesc.getParameters().get(0);
      QName typeQName = paramDesc.getTypeQName();
      Class reqStructType = paramDesc.getJavaType();
      if (reqStructType == null)
         throw new IllegalArgumentException("Cannot obtain java type for: " + typeQName); 
      
      log.debug("wrapRequestParameters: " + reqStructType.getName());
      try
      {
         List varNames = paramDesc.getWrappedVariables();
         Object reqStruct = reqStructType.newInstance();
         for (int i = 0; i < varNames.size(); i++)
         {
            String varName = (String)varNames.get(i);
            PropertyDescriptor pd = new PropertyDescriptor(varName, reqStructType);
            Method method = pd.getWriteMethod();
            Object value = inParams[i];
            log.debug(" " + method.getName() + ": " + (value != null ? value.getClass().getName() : null));
            method.invoke(reqStruct, new Object[]{value});
         }
         
         return reqStruct;
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception e)
      {
         throw new IllegalStateException("Cannot wrap request structure: " + e);
      }
   }

   public static Object[] unwrapRequestParameters(OperationDesc opDesc, Object reqStruct)
   {
      assertOperationDesc(opDesc);

      if (reqStruct == null)
         throw new IllegalArgumentException("Request struct cannot be null");

      ParameterDesc paramDesc = (ParameterDesc)opDesc.getParameters().get(0);
      List varNames = paramDesc.getWrappedVariables();
      Class reqStructType = reqStruct.getClass();
      
      log.debug("unwrapRequestParameters: " + reqStructType.getName());
      Object[] inParams = new Object[varNames.size()];
      try
      {
         for (int i = 0; i < varNames.size(); i++)
         {
            String varName = (String)varNames.get(i);
            PropertyDescriptor pd = new PropertyDescriptor(varName, reqStructType);
            Method method = pd.getReadMethod();
            Object value = method.invoke(reqStruct, new Object[]{});
            log.debug(" " + method.getName() + ": " + (value != null ? value.getClass().getName() : null));
            inParams[i] = value;
         }
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException("Cannot unwrap request structure: " + e);
      }

      return inParams;
   }

   public static Object wrapResponseParameter(OperationDesc opDesc, Object outParam)
   {
      assertOperationDesc(opDesc);

      ParameterDesc paramDesc = opDesc.getReturnParamDesc();
      QName typeQName = paramDesc.getTypeQName();
      Class resStructType = paramDesc.getJavaType();
      if (resStructType == null)
         throw new IllegalArgumentException("Cannot obtain java type for: " + typeQName); 
      
      log.debug("wrapResponseParameter: " + resStructType.getName());
      try
      {
         Object resStruct = resStructType.newInstance();
         List varNames = paramDesc.getWrappedVariables();
         if (varNames.size() > 0)
         {
            String varName = (String)varNames.get(0);
            PropertyDescriptor pd = new PropertyDescriptor(varName, resStructType);
            Method method = pd.getWriteMethod();
            Object value = outParam;
            log.debug(" " + method.getName() + ": " + (value != null ? value.getClass().getName() : null));
            method.invoke(resStruct, new Object[] { value });
         }
         return resStruct;
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception e)
      {
         throw new IllegalStateException("Cannot wrap response structure: " + e);
      }
   }

   public static Object unwrapResponseParameter(OperationDesc opDesc, Object resStruct)
   {
      assertOperationDesc(opDesc);

      if (resStruct == null)
         throw new IllegalArgumentException("Response struct cannot be null");

      ParameterDesc paramDesc = opDesc.getReturnParamDesc();
      List varNames = paramDesc.getWrappedVariables();
      Class resStructType = resStruct.getClass();

      Object retValue = null;
      log.debug("unwrapResponseParameter: " + resStructType.getName());
      try
      {
         if (varNames.size() > 0)
         {
            String varName = (String)varNames.get(0);
            PropertyDescriptor pd = new PropertyDescriptor(varName, resStructType);
            Method method = pd.getReadMethod();
            Object value = method.invoke(resStruct, new Object[] {});
            log.debug(" " + method.getName() + ": " + (value != null ? value.getClass().getName() : null));
            retValue = value;
         }
      }
      catch (RuntimeException rte)
      {
         throw rte;
      }
      catch (Exception e)
      {
         throw new IllegalArgumentException("Cannot unwrap request structure: " + e);
      }

      return retValue;
   }

   private static void assertOperationDesc(OperationDesc opDesc)
   {
      if (opDesc.getStyle() != Style.DOCUMENT)
         throw new IllegalStateException("Unexpected style: " + opDesc);

      if (opDesc.getParameters().size() != 1)
         throw new IllegalStateException("Unexpected number of input parameters for: " + opDesc);

      if (opDesc.isWrapParameters() == false)
         throw new IllegalStateException("Operation meta data not marked for wrapping: " + opDesc);
      
      ParameterDesc paramDesc = (ParameterDesc)opDesc.getParameters().get(0);
      if (paramDesc.getWrappedVariables() == null)
         throw new IllegalStateException("Parameter meta data does not contain a variable name list: " + paramDesc);
      
      if (opDesc.isOneWay() == false)
      {
         if (opDesc.getReturnParamDesc() == null)
            throw new IllegalStateException("Unexpected null return parameter: " + opDesc);

         ParameterDesc retMetaData = opDesc.getReturnParamDesc();
         if (retMetaData.getWrappedVariables() == null)
            throw new IllegalStateException("Return parameter meta data does not contain a variable name list: " + retMetaData);
      }
   }
}
