/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ejb3.test.mdb.unit;

import javax.jms.MessageProducer;
import javax.jms.DeliveryMode;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import org.jboss.ejb3.test.mdb.Stateless;
import org.jboss.ejb3.test.mdb.TestStatus;
import org.jboss.logging.Logger;
import org.jboss.security.SecurityAssociation;
import org.jboss.security.SimplePrincipal;
import org.jboss.test.JBossTestCase;
import junit.framework.Test;

/**
 * Sample client for the jboss container.
 *
 * @author <a href="mailto:bill@burkecentral.com">Bill Burke</a>
 * @version $Id$
 */

public class MDBUnitTestCase
        extends JBossTestCase
{
   private static final Logger log = Logger
   .getLogger(MDBUnitTestCase.class);

   static boolean deployed = false;
   static int test = 0;

   public MDBUnitTestCase(String name)
   {

      super(name);

   }
   
   public void testOverrideQueue() throws Exception
   {
      SecurityAssociation.setPrincipal(new SimplePrincipal("anyone"));
      SecurityAssociation.setCredential("password".toCharArray());
      
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      QueueConnection cnn = null;
      QueueSender sender = null;
      QueueSession session = null;

      Queue queue = (Queue) getInitialContext().lookup("queue/overridequeuetest");
      QueueConnectionFactory factory = (QueueConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createQueueConnection();
      session = cnn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");

      sender = session.createSender(queue);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.overrideQueueFired());
   }
   
   public void testNondurableQueue() throws Exception
   {
      SecurityAssociation.setPrincipal(new SimplePrincipal("anyone"));
      SecurityAssociation.setCredential("password".toCharArray());
      
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      QueueConnection cnn = null;
      QueueSender sender = null;
      QueueSession session = null;

      Queue queue = (Queue) getInitialContext().lookup("queue/nondurablemdbtest");
      QueueConnectionFactory factory = (QueueConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createQueueConnection();
      session = cnn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");
      msg.setJMSDeliveryMode(DeliveryMode.NON_PERSISTENT);

      sender = session.createSender(queue);
      sender.setDeliveryMode(DeliveryMode.NON_PERSISTENT);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.nondurableQueueFired());
   }
   
   public void testDefaultedQueue() throws Exception
   {
      SecurityAssociation.setPrincipal(new SimplePrincipal("anyone"));
      SecurityAssociation.setCredential("password".toCharArray());
      
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      QueueConnection cnn = null;
      QueueSender sender = null;
      QueueSession session = null;

      Queue queue = (Queue) getInitialContext().lookup("queue/defaultedmdbtest");
      QueueConnectionFactory factory = (QueueConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createQueueConnection();
      session = cnn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");

      sender = session.createSender(queue);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.defaultedQueueFired());
   }

   public void testQueue() throws Exception
   {
      SecurityAssociation.setPrincipal(new SimplePrincipal("anyone"));
      SecurityAssociation.setCredential("password".toCharArray());
      
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      QueueConnection cnn = null;
      QueueSender sender = null;
      QueueSession session = null;

      Queue queue = (Queue) getInitialContext().lookup("queue/mdbtest");
      QueueConnectionFactory factory = (QueueConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createQueueConnection();
      session = cnn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");

      sender = session.createSender(queue);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.queueFired());
      assertTrue(status.interceptedQueue());
      assertTrue(status.postConstruct());

      //TODO: Figure out how to test preDestroy gets invoked
      //assertTrue(status.preDestroy());
      
      Stateless stateless = (Stateless)getInitialContext().lookup("Stateless");
      assertNotNull(stateless);
      String state = stateless.getState();
      assertEquals("Set", state);
   }

   public void testTopic() throws Exception
   {
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      TopicConnection cnn = null;
      MessageProducer sender = null;
      TopicSession session = null;

      Topic topic = (Topic) getInitialContext().lookup("topic/mdbtest");
      TopicConnectionFactory factory = (TopicConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createTopicConnection();
      session = cnn.createTopicSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");

      sender = session.createProducer(topic);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.topicFired());
      assertTrue(status.interceptedTopic());
      assertFalse(status.postConstruct());
      assertFalse(status.preDestroy());
   } 
   
   public void testRuntimeException() throws Exception
   {
      TestStatus status = (TestStatus) getInitialContext().lookup("TestStatusBean/remote");
      status.clear();
      QueueConnection cnn = null;
      QueueSender sender = null;
      QueueSession session = null;

      Queue queue = (Queue) getInitialContext().lookup("queue/bmtmdbtest");
      QueueConnectionFactory factory = (QueueConnectionFactory) getInitialContext().lookup("ConnectionFactory");
      cnn = factory.createQueueConnection();
      session = cnn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      TextMessage msg = session.createTextMessage("Hello World");

      sender = session.createSender(queue);
      sender.send(msg);

      Thread.sleep(1000);
      assertEquals(1, status.bmtQueueRan());
   }

   public static Test suite() throws Exception
   {
      return getDeploySetup(MDBUnitTestCase.class, "mdbtest-service.xml, mdb-test.jar");
   }

}
