/*
 * JBoss, the OpenSource WebOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.web.loadbalancer.scheduler;

import java.net.URL;

import org.jboss.mx.util.JBossNotificationBroadcasterSupport;
import org.jboss.web.loadbalancer.util.Constants;

/**
 * A class that holds information about a target node.
 *
 * @jmx:mbean name="jboss.web.loadbalancer: type=Host"
 * @author Thomas Peuss <jboss@peuss.de>
 * @version $Revision:1$
 */
public class Host extends JBossNotificationBroadcasterSupport implements HostMBean
{
  private URL url;
  private int lbFactor;
  private int state=Constants.STATE_NODE_UP;
  private HostStatistics statistics=new HostStatistics();

  public Host(URL url)
  {
    this.url=url;
  }

  public String toString()
  {
    return "[URL=" + this.getUrl().toExternalForm() + ", State=" + this.getStateString() +
        ", LbFactor=" + this.getLbFactor() + ", Stats=" + this.getStatistics() + "]";
  }

  public int hashCode()
  {
    return url.hashCode();
  }

  public boolean equals(Object obj)
  {
    if (obj==null)
    {
      return false;
    }

    if (!(obj instanceof Host))
    {
      return false;
    }

    return (this.hashCode()==obj.hashCode());
  }

  public void addRequest(int responseTime)
  {
    statistics.addRequest(responseTime);
  }

  public void incCurrentConnections()
  {
    statistics.incCurrentConnections();
  }

  public void decCurrentConnections()
  {
    statistics.decCurrentConnections();
  }

  /**
   * @jmx:managed-attribute
   */
  public int getCurrentConnections()
  {
    return statistics.getCurrentConnections();
  }

  /**
   * @jmx:managed-attribute
   */
  public URL getUrl()
  {
    return url;
  }

  public void setUrl(URL url)
  {
    this.url = url;
  }

  /**
   * @jmx:managed-attribute
   */
  public int getLbFactor()
  {
    return lbFactor;
  }

  /**
   * @jmx:managed-attribute
   */
  public void setLbFactor(int lbFactor)
  {
    this.lbFactor = lbFactor;
  }

  /**
   * @jmx:managed-attribute
   */
  public HostStatistics getStatistics()
  {
    return statistics;
  }

  public void setStatistics(HostStatistics statistics)
  {
    this.statistics = statistics;
  }


  /**
   * @jmx:managed-attribute
   * @return
   */
  public int getState()
  {
    return state;
  }

  public void setState(int state)
  {
    this.state = state;
    this.sendNotification(new HostStateChangedNotification(this, "Host: "+this+". State changed to "+this.getStateString()));
  }

  /**
   * @jmx:managed-attribute
   */
  public String getStateString()
  {
    return Constants.STATE_STRINGS[state];
  }

  /**
   * @jmx:managed-operation
   */
  public void markNodeUp()
  {
    this.setState(Constants.STATE_NODE_UP);
  }

  /**
   * @jmx:managed-operation
   */
  public void markNodeDown()
  {
    this.setState(Constants.STATE_NODE_DOWN);
  }

  /**
   * @jmx:managed-operation
   */
  public void markNodeForcedDown()
  {
    this.setState(Constants.STATE_NODE_FORCED_DOWN);
  }

  /**
   * @jmx:managed-operation
   */
  public void resetStatistics()
  {
    statistics.reset();
  }
}
