/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/
 
package org.jboss.monitor.services;

import javax.management.Notification;
import javax.management.ObjectName;

import EDU.oswego.cs.dl.util.concurrent.SynchronizedLong;

import org.jboss.system.ListenerServiceMBeanSupport;

/**
 * A simple JMX notification listener that outputs notifications as log.INFO
 * messages, and demonstrates the usefulness of ListenerServiceMBeanSupport.
 *
 * @jmx:mbean
 *    extends="org.jboss.system.ListenerServiceMBean"
 * 
 * @author  <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 *
 * @version $Revision:1$
**/
public class NotificationListener
    extends ListenerServiceMBeanSupport
    implements NotificationListenerMBean
{
   // Private Data --------------------------------------------------
    
   /** Number of processed JMX notifications */
   private SynchronizedLong notificationCount;
   
   /** Dynamic subscriptions flag */
   private boolean dynamicSubscriptions;
   
   /** Listener MBean */
   private ObjectName notificationListener;
    
   // Constructors --------------------------------------------------
   
   /**
    * CTOR
   **/
   public NotificationListener()
   {
      this.notificationCount = new SynchronizedLong(0);
      this.dynamicSubscriptions = true;
      this.notificationListener = null;
   }
   
   // NotificationListenerMBean Implementation -----------------------
   
   /**
    * Number of notifications received.
    *
    * @jmx:managed-attribute
   **/
   public long getNotificationCount()
   {
      return this.notificationCount.get();
   }

   /**
    * Enables/disables dynamic subscriptions
    *
    * @jmx:managed-attribute
   **/
   public void setDynamicSubscriptions(boolean dynamicSubscriptions)
   {
      this.dynamicSubscriptions = dynamicSubscriptions;
   }

   /**
    * Gets the dynamic subscriptions status
    *
    * @jmx:managed-attribute
   **/
   public boolean getDynamicSubscriptions()
   {
      return this.dynamicSubscriptions;
   }
   
   /**
    * Sets listener of notifications
    *
    * @jmx:managed-attribute
   **/
   public void setNotificationListener(ObjectName notificationListener)
   {
      this.notificationListener = notificationListener;
   }
   
   /**
    * Gets listener of notifications
    *
    * @jmx:managed-attribute
   **/
   public ObjectName getNotificationListener()
   {
      return this.notificationListener;
   }   
   
   // Lifecycle control (ServiceMBeanSupport) -----------------------
   
   /**
    * Start 
   **/
   public void startService()
      throws Exception
   {
      if (this.notificationListener == null)
         super.subscribe(this.dynamicSubscriptions); // listener is me!
      else
         super.subscribe(this.dynamicSubscriptions, this.notificationListener);
   }
   
   /**
    * Stop
   **/
   public void stopService()
      throws Exception
   {
      // unsubscribe for notifications
      super.unsubscribe();
   }
   
   /**
    * Overriden to add handling!
   **/
   public void handleNotification2(Notification notification, Object handback)
   {
      log.info("Got notification (#" + Long.toString(this.notificationCount.increment())
             + "): " + notification + ", handback: " + handback);
   }
}
