/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.net.ssl;

import java.io.IOException;
import java.security.KeyStore;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.net.ssl.TrustManager;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.KeyManagerFactory;

import org.jboss.security.SecurityDomain;
import org.apache.tomcat.util.net.jsse.JSSE14SocketFactory;

/**
 * Extends the tomcat JSSE14SocketFactory to obtain the server key and trust
 * stores from the SecurityDomain defined by the securityDomain attribute
 * of the connector.
 * 
 */ 
public class JBossSocketFactory
   extends JSSE14SocketFactory
{
   private SecurityDomain securityDomain;

   public JBossSocketFactory()
   {
   }

   public void setAttribute(String name, Object value)
   {
      if (name.equalsIgnoreCase("securityDomain"))
      {
         try
         {
            setSecurityDomainName((String) value);
         }
         catch (Exception e)
         {
            IllegalArgumentException ex =
               new IllegalArgumentException("Failed to set security domain");
            ex.initCause(e);
            throw ex;
         }
      }
      super.setAttribute(name, value);
   }

   /**
    * Set the SecurityDomain to use for the key/trust stores
    * 
    * @param jndiName - the jndi name of the SecurityDomain binding
    * @throws NamingException
    * @throws IOException
    */ 
   public void setSecurityDomainName(String jndiName)
      throws NamingException, IOException
   {
      InitialContext iniCtx = new InitialContext();
      securityDomain = (SecurityDomain) iniCtx.lookup(jndiName);
   }

   /**
    * Gets the SSL server's keystore from the SecurityDomain.
    * 
    * @param type - ignored, this comes from the security domain config
    * @param pass - ignore, this comes from the security domain config
    * @return the KeyStore for the server cert
    * @throws IOException
    */ 
   protected KeyStore getKeystore(String type, String pass)
      throws IOException
   {
      return securityDomain.getKeyStore();
   }

   /*
    * Gets the SSL server's truststore from the SecurityDomain.
    
    * @param type - ignored, this comes from the security domain config
    * @return the KeyStore for the trusted signers store
    */
   protected KeyStore getTrustStore(String type) throws IOException
   {
      return securityDomain.getTrustStore();
   }

   /**
    * Override to obtain the TrustManagers from the security domain.
    * 
    * @param keystoreType - ignored, this comes from the security domain
    * @param algorithm - ignored, this comes from the security domain
    * @return the array of TrustManagers from the security domain
    * @throws Exception
    */ 
   protected TrustManager[] getTrustManagers(String keystoreType, String algorithm)
      throws Exception
   {
      TrustManagerFactory tmf = securityDomain.getTrustManagerFactory();
      TrustManager[] trustMgrs = null;

      if( tmf != null )
      {
          trustMgrs = tmf.getTrustManagers();
      }
      return trustMgrs;
   }

   /**
    * Override to obtain the KeyManagers from the security domain.
    * 
    * @param keystoreType - ignored, this comes from the security domain
    * @param algorithm - ignored, this comes from the security domain
    * @param keyAlias - ignored
    * @return the array of KeyManagers from the security domain
    * @throws Exception
    */ 
   protected KeyManager[] getKeyManagers(String keystoreType, String algorithm,
      String keyAlias)
      throws Exception
   {
      KeyManagerFactory kmf = securityDomain.getKeyManagerFactory();
      KeyManager[] keyMgrs = null;
      if( kmf != null )
      {
         keyMgrs = kmf.getKeyManagers();
      }
      return keyMgrs;
   }
}
