/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.cache.test.standAloneAop;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.aop.TreeCacheAop;
import org.jboss.logging.Logger;

/**
 * Test object graph handling in aop, e.g., circular reference, multiple reference, link, etc.
 * @author Ben Wang  
 */

public class ReplicatedObjectGraphAopTest extends TestCase
{
   Logger log = Logger.getLogger(ReplicatedObjectGraphAopTest.class);
   TreeCacheAopTester tester1;
   TreeCacheAopTester tester2;


   public ReplicatedObjectGraphAopTest(String name)
   {
      super(name);
   }

   protected void setUp() throws Exception
   {
      super.setUp();
      log.info("setUp() ....");
      String configFile = "META-INF/replSync-service.xml";
      tester1 = new TreeCacheAopTester(configFile);
      tester2 = new TreeCacheAopTester(configFile);
   }

   protected void tearDown() throws Exception
   {
      super.tearDown();
      tester1.stop();
      tester1 = null;
      tester2.stop();
      tester2 = null;
   }

//   public void testDummy() {}

   public void testSetup()
   {
      log.info("testSetup() ....");
      try {
         tester1.testSetup();
         tester2.testSetup();
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("testSetup(): fails. " + ex.toString());
      }
   }


   private void stage1()
   {
      tester1.createPerson("/person/joe", "Joe Black", 31);
      Person joe = (Person) tester1.getPerson("/person/joe");
      tester1.createPerson("/person/ben", "Ben Hogan", 51);
      Person ben = (Person) tester1.getPerson("/person/ben");

      Address addr = tester1.createAddress();
      addr.setStreet("123 Albert Ave.");
      addr.setCity("Sunnyvale");
      addr.setZip(94087);

      // They share the sub-object: address
      joe.setAddress(addr);
      assertEquals("Joe's address should still be valid ", "Sunnyvale", tester1.getCity("/person/joe"));
      ben.setAddress(addr);
      assertEquals("Ben's address should still be valid ", "Sunnyvale", tester1.getCity("/person/ben"));
   }

   private void stage2(TreeCacheAopTester tester)
   {
      //
      tester.removePerson("/person/joe");
      log.info("stage2(): " + tester.printCacheDetails());
      assertEquals("Ben's address should still be valid ", "Sunnyvale", tester.getCity("/person/ben"));
      Person ben = (Person) tester.getPerson("/person/ben");
      Address addr = ben.getAddress();
      addr.setCity("Santa Clara");
      assertEquals("Ben's address should be changed ", "Santa Clara", tester.getCity("/person/ben"));
   }

   public void testRefCountCheckRepl() throws Exception
   {
      log.info("testRefCountCheckRepl() ...");
      stage1();
      assertEquals("Ben and Joe's address should be the same ", tester1.getCity("/person/joe"),
            tester1.getCity("/person/ben"));
      stage2(tester2);
      assertEquals("Ben's address should be changed on tester1 as well ", "Santa Clara", tester1.getCity("/person/ben"));
      tester2.removePerson("/person/ben");
      log.info("testRefCountCheckRepl(): tester1 " + tester1.printCacheDetails());
      log.info("testRefCountCheckRepl(): tester2 " + tester2.printCacheDetails());
   }

   public void testCircularReference1() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference1() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      cache1.putObject("/link/parent", parent);
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
//        tester1.getCache().putObject("/link/parent/child", child);
      log.info("testCircularReference1(): tester1 " + tester1.printCacheDetails());
      TreeCacheAop cache2 = tester2.getCache();
      assertEquals("parent", ((Link) cache2.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache2.getObject("/link/parent")).getLink().getName());
      log.info("testCircularReference1(): tester2 " + tester2.printCacheDetails());
      ((Link) cache2.getObject("/link/parent")).setLink(null);
      assertNull("Child should be null", ((Link) cache2.getObject("/link/parent")).getLink());
   }

   public void testCircularReference2() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference2() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      tester1.getCache().putObject("/link/parent", parent);
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
      log.info("testCircularReference2(): tester1 " + tester1.printCacheDetails());
      TreeCacheAop cache2 = tester2.getCache();
      assertEquals("parent", ((Link) cache2.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache2.getObject("/link/parent")).getLink().getName());
      log.info("testCircularReference2(): tester2 " + tester2.printCacheDetails());
      ((Link) cache2.getObject("/link/parent")).setLink(null);
      assertNull("Child should be null", ((Link) cache2.getObject("/link/parent")).getLink());
   }

   public void testCircularReference3() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference3() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      tester1.getCache().putObject("/link/parent", parent);
      tester1.getCache().putObject("/link/child", child);
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
      assertEquals("child", ((Link) cache1.getObject("/link/child")).getName());
      assertEquals("parent", ((Link) cache1.getObject("/link/child")).getLink().getName());
//        tester1.getCache().putObject("/link/parent/child", child);
      log.info("testCircularReference3(): tester1 " + tester1.printCacheDetails());
      TreeCacheAop cache2 = tester2.getCache();
      assertEquals("parent", ((Link) cache2.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache2.getObject("/link/parent")).getLink().getName());
      assertEquals("child", ((Link) cache2.getObject("/link/child")).getName());
      assertEquals("parent", ((Link) cache2.getObject("/link/child")).getLink().getName());
      log.info("testCircularReference3(): tester2 " + tester2.printCacheDetails());
      ((Link) cache2.getObject("/link/parent")).setLink(null);
      assertNull("Child should be null", ((Link) cache2.getObject("/link/parent")).getLink());
   }

   public static Test suite() throws Exception
   {
      return new TestSuite(ReplicatedObjectGraphAopTest.class);
   }

   public static void main(String[] args) throws Exception
   {
      junit.textui.TestRunner.run(suite());
   }

}

