/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.cache.test.standAloneAop;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.aop.TreeCacheAop;
import org.jboss.logging.Logger;

/**
 * Test object graph handling in aop, e.g., circular reference, multiple reference, link, etc.
 * @author Ben Wang
 */

public class ObjectGraphAopTest extends TestCase
{
   Logger log = Logger.getLogger(ObjectGraphAopTest.class);
   TreeCacheAopTester tester1;

   public ObjectGraphAopTest(String name)
   {
      super(name);
   }

   protected void setUp() throws Exception
   {
      super.setUp();
      log.info("setUp() ....");
      String configFile = "META-INF/local-service.xml";
      tester1 = new TreeCacheAopTester(configFile);
   }

   protected void tearDown() throws Exception
   {
      super.tearDown();
      tester1.stop();
      tester1 = null;
   }

//   public void testDummy() {}

   public void testSetup()
   {
      log.info("testSetup() ....");
      try {
         tester1.testSetup();
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("testSetup(): fails. " + ex.toString());
      }
   }

   /**
    * Test shared sub-object. In diagram, it is a forest
    * where the sub-object is connected by two tree parents.
    */
   public void testMultipleReference() throws Exception
   {
      log.info("testMultipleReference() ...");
      tester1.createPerson("/person/joe", "Joe Black", 31);
      Person joe = (Person) tester1.getPerson("/person/joe");
      tester1.createPerson("/person/ben", "Ben Hogan", 51);
      Person ben = (Person) tester1.getPerson("/person/ben");

      Address addr = tester1.createAddress();
      addr.setStreet("123 Albert Ave.");
      addr.setCity("Sunnyvale");
      addr.setZip(94087);

      // They share the sub-object: address
      log.info("testMultipleReference(): set Joe address");
      joe.setAddress(addr);
      log.info("testMultipleReference(): set Ben address");
      ben.setAddress(addr);

      log.info("testMultipleReference(): verify");
      assertEquals(tester1.getCity("/person/joe"), tester1.getCity("/person/ben"));
      addr.setCity("Santa Clara");
      assertEquals(tester1.getCity("/person/joe"), tester1.getCity("/person/ben"));
   }

   public void testRefCount() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testRefCount() ...");
      stage1();
      stage2(tester1);

      tester1.removePerson("/person/ben");
      log.info("stage2(): " + tester1.printCacheDetails());
   }

   private void stage1()
   {
      tester1.createPerson("/person/joe", "Joe Black", 31);
      Person joe = (Person) tester1.getPerson("/person/joe");
      tester1.createPerson("/person/ben", "Ben Hogan", 51);
      Person ben = (Person) tester1.getPerson("/person/ben");

      Address addr = tester1.createAddress();
      addr.setStreet("123 Albert Ave.");
      addr.setCity("Sunnyvale");
      addr.setZip(94087);

      // They share the sub-object: address
      joe.setAddress(addr);
      assertEquals("Joe's address should still be valid ", "Sunnyvale", tester1.getCity("/person/joe"));
      ben.setAddress(addr);
      assertEquals("Ben's address should still be valid ", "Sunnyvale", tester1.getCity("/person/ben"));
   }

   private void stage2(TreeCacheAopTester tester)
   {
      //
      tester.removePerson("/person/joe");
      log.info("stage2(): " + tester.printCacheDetails());
      assertEquals("Ben's address should still be valid ", "Sunnyvale", tester.getCity("/person/ben"));
      Person ben = (Person) tester.getPerson("/person/ben");
      Address addr = ben.getAddress();
      addr.setCity("Santa Clara");
      assertEquals("Ben's address should be changed ", "Santa Clara", tester.getCity("/person/ben"));
   }

   public void testCircularReference1() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference1() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      cache1.putObject("/link/parent", parent);
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
//        tester1.getCache().putObject("/link/parent/child", child);
      log.info("testCircularReference1(): tester1 " + tester1.printCacheDetails());
   }

   public void testCircularReference2() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference2() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      tester1.getCache().putObject("/link/parent", parent);
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
      log.info("testCircularReference2(): tester1 " + tester1.printCacheDetails());
   }

   public void testCircularReference3() throws Exception
   {
//        try {Thread.sleep(10000); } catch (Exception e) {};
      log.info("testCircularReference3() ...");
      Link parent = new Link("parent");
      Link child = new Link("child");
      tester1.getCache().putObject("/link/parent", parent);
      tester1.getCache().putObject("/link/child", child);
      parent.setLink(child);
      child.setLink(parent);
      TreeCacheAop cache1 = tester1.getCache();
      assertEquals("parent", ((Link) cache1.getObject("/link/parent")).getName());
      assertEquals("child", ((Link) cache1.getObject("/link/parent")).getLink().getName());
      assertEquals("child", ((Link) cache1.getObject("/link/child")).getName());
      assertEquals("parent", ((Link) cache1.getObject("/link/child")).getLink().getName());
//        tester1.getCache().putObject("/link/parent/child", child);
      log.info("testCircularReference3(): tester1 " + tester1.printCacheDetails());
   }

   public void testObjectIdentity() throws Exception
   {

   }

   public static Test suite() throws Exception
   {
      return new TestSuite(ObjectGraphAopTest.class);
   }

   public static void main(String[] args) throws Exception
   {
      junit.textui.TestRunner.run(suite());
   }

}

