/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.cache.test.eviction;

import junit.framework.TestCase;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.jboss.cache.Fqn;
import org.jboss.cache.eviction.Region;
import org.jboss.cache.eviction.RegionManager;
import org.jboss.cache.eviction.LRUAlgorithm;
import org.jboss.cache.eviction.EvictionException;

import java.util.HashMap;

/**
 * @author Ben Wang, Feb 11, 2004
 */
public class LRUAlgorithmUnitTestCase extends TestCase
{
   RegionManager regionManager_;
   LRUAlgorithm algo_;

   public LRUAlgorithmUnitTestCase(String s)
   {
      super(s);
   }

   public void setUp() throws Exception
   {
      super.setUp();
      algo_ = new LRUAlgorithm();
      DummyEvictionPolicy policy = new DummyEvictionPolicy();
      regionManager_ = new RegionManager(policy);
      regionManager_.createRegion("/a/b", algo_);
      /*
      try {
         Thread.sleep(10000);
      } catch (InterruptedException e) {
         e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
      }
      */
   }

   public void tearDown() throws Exception
   {
      super.tearDown();
   }

   /**
    * maxNodes = 1. Eception is evictFromCacheNode. Should be commented for now.
    */
   public void XtestEvictException() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);

      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 1, algo_.evictionQueueSize());

      region.setAddedNode(fqn2);
      region.setAddedNode(fqn3);

      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 1, algo_.evictionQueueSize());
   }


   /**
    * maxNodes = 0 case
    */
   public void testMaxNode1() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(0);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);

      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 2, algo_.evictionQueueSize());

   }

   /**
    * maxNodes = 1
    */
   public void testMaxNode2() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);

      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 1, algo_.evictionQueueSize());

      region.setAddedNode(fqn2);
      region.setAddedNode(fqn3);

      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 1, algo_.evictionQueueSize());
   }

   /**
   * TimeToIdleSeconds = 0
   */
   public void testIdleTimeSeconds1() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(0);
      region.setTimeToLiveSeconds(0);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);
      _sleep(2000);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size should be ", 2, algo_.evictionQueueSize());

   }

   /**
   * TimeToIdleSeconds = 1
   */
   public void testIdleTimeSeconds2() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(0);
      region.setTimeToLiveSeconds(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);
      region.setAddedNode(fqn3);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #1: ", 3, algo_.evictionQueueSize());
      _sleep(2000);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #2: ", 0, algo_.evictionQueueSize());
   }

   /**
   * TimeToIdleSeconds = 1 with node visited in between.
   */
   public void testIdleTimeSeconds3() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(0);
      region.setTimeToLiveSeconds(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);
      region.setAddedNode(fqn3);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #1: ", 3, algo_.evictionQueueSize());
      _sleep(2000);
      region.setVisitedNode(fqn2);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #2: ", 1, algo_.evictionQueueSize());
   }


   /**
    * Generic combo case.
    */
   public void testCombo1() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(2);
      region.setTimeToLiveSeconds(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #1: ", 2, algo_.evictionQueueSize());
      region.setAddedNode(fqn3);
      _sleep(2000);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #2: ", 1, algo_.evictionQueueSize());
   }

   /**
    * Generic combo case with newly added node should be around.
    */
   public void testCombo2() {
      Fqn fqn1 = Fqn.fromString("/a/b/c");
      Fqn fqn2 = Fqn.fromString("/a/b/d");
      Fqn fqn3 = Fqn.fromString("/a/b/e");
      Region region = regionManager_.getRegion("/a/b");
      region.setMaxNodes(2);
      region.setTimeToLiveSeconds(1);
      region.setAddedNode(fqn1);
      region.setAddedNode(fqn2);
      region.setRemovedNode(fqn2);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #1: ", 1, algo_.evictionQueueSize());
      region.setAddedNode(fqn3);
      _sleep(2000);
      try {
         algo_.process(region);
      } catch (EvictionException e) {
         fail("testMaxNode: process failed " +e);
         e.printStackTrace();
      }
      assertEquals("Queue size #2: ", 1, algo_.evictionQueueSize());
   }

   void _sleep(long msecs) {
      try {
         Thread.sleep(msecs);
      } catch (InterruptedException e) {
         e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
      }
   }

   void log(String msg)
   {
      System.out.println("-- " + msg);
   }

   public static Test suite()
   {
      return new TestSuite(LRUAlgorithmUnitTestCase.class);
   }

   public static void main(String[] args)
   {
      junit.textui.TestRunner.run(suite());
   }

}
