/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.test.cache.test.eviction;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.Fqn;
import org.jboss.cache.PropertyConfigurator;
import org.jboss.cache.aop.AOPInstance;
import org.jboss.cache.aop.TreeCacheAop;

/**
 * @author Ben Wang, Feb 11, 2004
 */
public class AopLRUPolicyUnitTestCase extends TestCase
{
   TreeCacheAop cache_;
   int wakeupIntervalMillis_ = 0;

   public AopLRUPolicyUnitTestCase(String s)
   {
      super(s);
   }

   public void setUp() throws Exception
   {
      super.setUp();
      initCaches();
      wakeupIntervalMillis_ = cache_.getEvictionThreadWakeupIntervalSeconds() *1000;
      log("wakeupInterval is " +wakeupIntervalMillis_);
      if(wakeupIntervalMillis_ <=0)
         fail("testEviction(): eviction thread wake up interval is illegal " +wakeupIntervalMillis_);

   }

   void initCaches() throws Exception
   {
//      _sleep(10000);
      cache_ = new TreeCacheAop();
      PropertyConfigurator config = new PropertyConfigurator();
      config.configure(cache_, "META-INF/local-aop-eviction-service.xml"); // read in generic local xml
      cache_.setTransactionManagerLookupClass("org.jboss.cache.DummyTransactionManagerLookup");
      cache_.startService();
   }

   public void tearDown() throws Exception
   {
      super.tearDown();
      cache_.stopService();
   }

   public void testSimpleEviction() {
      String rootStr = "/aop/";
      AOPInstance aop = new AOPInstance();
      try {
         for(int i=0; i < 4; i++) {
            String stri = rootStr +i;
            Fqn fqni = Fqn.fromString(stri);
            cache_.put(fqni, stri, stri);
            cache_.put(fqni, AOPInstance.KEY, aop);   // signals that this is an aop node.
            cache_.put(fqni, TreeCacheAop.CLASS_INTERNAL, String.class);   // signals that this is an aop node.
            for(int j=0; j < 3; j++) {
               String strj = stri +"/" +j;
               Fqn fqnj = Fqn.fromString(strj);
               cache_.put(fqnj, strj, strj);
            }
         }
      } catch (Exception e) {
         e.printStackTrace();
         fail("Failed to insert data" +e);
      }

      int period = (wakeupIntervalMillis_ +1000);
      log("period is " +period);
      _sleep(period);  // it really depends on the eviction thread time.

      try {
         String str = rootStr + "0";
         String val = (String)cache_.get(Fqn.fromString(str), str);
         assertNull("Node should be empty ", val);
         str = rootStr + "3";
         val = (String)cache_.get(Fqn.fromString(str), str);
         assertNotNull("Node should not be empty if maxElements is 4 ", val);
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

   public void testUpdateEviction() {
      String rootStr = "/aop/";
      AOPInstance aop = new AOPInstance();
      try {
         for(int i=0; i < 4; i++) {
            String stri = rootStr +i;
            Fqn fqni = Fqn.fromString(stri);
            cache_.put(fqni, stri, stri);
            cache_.put(fqni, AOPInstance.KEY, aop);   // signals that this is an aop node.
            cache_.put(fqni, TreeCacheAop.CLASS_INTERNAL, String.class);   // signals that this is an aop node.
            for(int j=0; j < 2; j++) {
               String strj = stri +"/" +j;
               Fqn fqnj = Fqn.fromString(strj);
               cache_.put(fqnj, strj, strj);
            }
         }
      } catch (Exception e) {
         e.printStackTrace();
         fail("Failed to insert data" +e);
      }

      int period = (wakeupIntervalMillis_ +1000);
      log("period is " +period);
      _sleep(period);  // it really depends on the eviction thread time.
      String str = rootStr + "3";
      Fqn fqn = Fqn.fromString(str);
      try {
         cache_.get(fqn, str);   // just to keep it fresh
         _sleep(period);  // it really depends the eviction thread time.
         cache_.get(fqn, str);   // just to keep it fresh
         _sleep(period);  // it really depends the eviction thread time.
         String val = (String)cache_.get(rootStr +"3/1", rootStr+"3/1");
         assertNotNull("Node should not be empty ", val);
         _sleep(wakeupIntervalMillis_*2 +2000);
         val = (String)cache_.get(rootStr +"3", rootStr+"3");
         assertNull("Node should be empty. But this is broken because of TreeCache._removeData() has disabled " +
                    "sendNodeEvent. See the FIXME. ", val);
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

   public void testRemoveEviction() {
      String rootStr = "/aop/";
      AOPInstance aop = new AOPInstance();
      try {
         for(int i=0; i < 4; i++) {
            String stri = rootStr +i;
            Fqn fqni = Fqn.fromString(stri);
            cache_.put(fqni, stri, stri);
            cache_.put(fqni, AOPInstance.KEY, aop);   // signals that this is an aop node.
            for(int j=0; j < 2; j++) {
               String strj = stri +"/" +j;
               Fqn fqnj = Fqn.fromString(strj);
               cache_.put(fqnj, strj, strj);
            }
         }
      } catch (Exception e) {
         e.printStackTrace();
         fail("Failed to insert data" +e);
      }

      int period = (wakeupIntervalMillis_ +1000);
      log("period is " +period);
      _sleep(period);  // it really depends on the eviction thread time.
      String str = rootStr + "3";
      Fqn fqn = Fqn.fromString(str);
      try {
         cache_.get(fqn, str);   // just to keep it fresh
         _sleep(period);  // it really depends the eviction thread time.
         cache_.get(fqn, str);   // just to keep it fresh
         _sleep(period);  // it really depends the eviction thread time.
         String val = (String)cache_.get(rootStr +"3/1", rootStr+"3/1");
         assertNotNull("Node should not be empty ", val);
         cache_.remove(rootStr +"3");
         val = (String)cache_.get(rootStr +"3", rootStr+"3");
         assertNull("Node should be empty ", val);
         _sleep(wakeupIntervalMillis_ +1000);
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

   void _sleep(long msecs) {
      try {
         Thread.sleep(msecs);
      } catch (InterruptedException e) {
         e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
      }
   }
   void log(String msg)
   {
      System.out.println("-- " + msg);
   }

   public static Test suite()
   {
      return new TestSuite(AopLRUPolicyUnitTestCase.class);
   }

   public static void main(String[] args)
   {
      junit.textui.TestRunner.run(suite());
   }

}
