/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.system.filterfactory;

import javax.management.NotificationFilter;
import javax.management.ObjectName;
import javax.management.relation.MBeanServerNotificationFilter;

import org.jboss.system.NotificationFilterFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Factory for MBeanServerNotificationFilter filters.
 * 
 * The produced filter is really meant for MBeanServerNotifications
 * emitted by the "JMImplementation:type=MBeanServerDelegate" MBean.  
 * 
 * filters-in zero or more Notification types, and zero or more
 * ObjectNames (contained in the MBeanServerNotification), so
 * you need to explicitly enable both the types and the ObjectNames.
 * 
 * In a nutshell, you'll be able to receive registration and/or unregistration
 * notifications for selected ObjectNames (i.e. MBean instances)!
 *
 * The passed filterConfig xml element fragment should look like:
 * 
 * <filter factory="MBeanServerNotificationFilterFactory">
 *   <enable type="JMX.mbean"/>
 *   ...
 *   <enable object-name="mydomain:type=mymbean"/>
 *   <enable object-name="mydomain:type=myothermbean"/>
 *   ...
 * </filter>
 * 
 * Note: JMX.mbean yields both notifications:
 * JMX.mbean.registered
 * JMX.mbean.unregistered 
 * 
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @version $Revision:1$
**/
public class MBeanServerNotificationFilterFactory
   implements NotificationFilterFactory
{
   // Constants -----------------------------------------------------
   
   /** the xml element and attribute supported by this factory */
   public static final String ENABLE_ELEMENT = "enable";
   public static final String ENABLE_TYPE_ATTRIBUTE = "type";
   public static final String ENABLE_OBJECTNAME_ATTRIBUTE = "object-name";
   
   /**
    * Default public CTOR (necessary)
    */
   public MBeanServerNotificationFilterFactory()
   {
      // empty
   }
   
   /**
    * The actual filter factory implementation
    */
   public NotificationFilter createNotificationFilter(Element filterConfig)
      throws Exception
   {
      // start off with a filter that does not allow any type/objectname
      MBeanServerNotificationFilter filter = new MBeanServerNotificationFilter();
      
      // filterConfig should point to the <filter factory="..."> element,
      // we are interested in its 'enable' children to configure the filter 
      NodeList filterChildren = filterConfig.getChildNodes();
      
      for (int i = 0; i < filterChildren.getLength(); i++) 
      {
         Node filterChildNode = filterChildren.item(i);
      
         // check if this is an 'enable' element, ignore everything else
         if (filterChildNode.getNodeName().equals(ENABLE_ELEMENT)) 
         {
            // look for 'type' attribute
            if (((Element)filterChildNode).hasAttribute(ENABLE_TYPE_ATTRIBUTE)) 
            {
               String type = ((Element)filterChildNode).getAttribute(ENABLE_TYPE_ATTRIBUTE);
               // enable this type in the filter
               filter.enableType(type);
            }
            else if (((Element)filterChildNode).hasAttribute(ENABLE_OBJECTNAME_ATTRIBUTE))
            {
               String objectName = ((Element)filterChildNode).getAttribute(ENABLE_OBJECTNAME_ATTRIBUTE);
               // enable this objectName in the filter
               // may throw MalformedObjectNameException
               filter.enableObjectName(new ObjectName(objectName));
            }
            else
            {
               throw new Exception("'" + ENABLE_ELEMENT + "' element must have a '"
                     + ENABLE_TYPE_ATTRIBUTE + "' or a '" + ENABLE_OBJECTNAME_ATTRIBUTE + "' attribute");
            }
         }
      }
      // we are done
      return filter;
   }
}
