/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.samples.callback;

import org.jboss.remoting.Client;
import org.jboss.remoting.HandleCallbackException;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerCallbackHandler;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.Callback;
import org.jboss.remoting.transport.Connector;

import java.util.Iterator;
import java.util.List;

/**
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class CallbackClient
{
   // Default locator values
   private static String transport = "socket";
   private static String host = "localhost";
   private static int port = 5400;

   private Client remotingClient;

   public void createRemotingClient(String locatorURI) throws Exception
   {
      InvokerLocator locator = new InvokerLocator(locatorURI);
      System.out.println("Calling remoting server with locator uri of: " + locatorURI);

      // This could have been new Client(locator), but want to show that subsystem param is null
      // Could have also been new Client(locator, "sample");
      remotingClient = new Client(locator, null);

   }

   public void makeInvocation() throws Throwable
   {
      Object response = remotingClient.invoke("Do something", null);
      System.out.println("Invocation response: " + response);
   }

   public void testPullCallback() throws Throwable
   {
      CallbackHandler callbackHandler = new CallbackHandler();
      // by passing only the callback handler, will indicate pull callbacks
      remotingClient.addListener(callbackHandler);
      // now make invocation on server, which should cause a callback to happen
      makeInvocation();

      List callbacks = remotingClient.getCallbacks();
      Iterator itr = callbacks.iterator();
      while(itr.hasNext())
      {
         Object obj = itr.next();
         if(obj instanceof Callback)
         {
            System.out.println("Callback value = " + ((Callback)obj).getCallbackObject());
         }
         else
         {
            System.out.println("Callback value = " + obj);
         }
      }

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);
   }

   public void testPushCallback() throws Throwable
   {
      // Need to create remoting server to receive callbacks.

      // Using loctor with port value one higher than the target server
      String callbackLocatorURI = transport + "://" + host + ":" + (port + 1);
      InvokerLocator callbackLocator = new InvokerLocator(callbackLocatorURI);

      // call to create remoting server to
      // receive client callbacks.
      setupServer(callbackLocator);

      CallbackHandler callbackHandler = new CallbackHandler();
      // Callback handle object will be passed back as part of the callback object
      String callbackHandleObject = "myCallbackHandleObject";
      // by passing only the callback handler, will indicate pull callbacks
      remotingClient.addListener(callbackHandler, callbackLocator, callbackHandleObject);
      // now make invocation on server, which should cause a callback to happen
      makeInvocation();

      // need to wait for brief moment so server can callback
      Thread.sleep(2000);

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);
   }

   public void setupServer(InvokerLocator locator) throws Exception
   {
      System.out.println("Starting remoting server with locator uri of: " + locator);
      Connector connector = new Connector();
      connector.setInvokerLocator(locator.getLocatorURI());
      connector.start();

      CallbackServer.SampleInvocationHandler invocationHandler = new CallbackServer.SampleInvocationHandler();
      // first parameter is sub-system name.  can be any String value.
      connector.addInvocationHandler("sample", invocationHandler);
   }


   /**
    * Can pass transport and port to be used as parameters.
    * Valid transports are 'rmi' and 'socket'.
    *
    * @param args
    */
   public static void main(String[] args)
   {
      if(args != null && args.length == 2)
      {
         transport = args[0];
         port = Integer.parseInt(args[1]);
      }
      String locatorURI = transport + "://" + host + ":" + port;
      CallbackClient client = new CallbackClient();
      try
      {
         client.createRemotingClient(locatorURI);

         // make call to test pull callbacks
         client.testPullCallback();

         // make call to test push callbacks
         client.testPushCallback();
      }
      catch(Throwable e)
      {
         e.printStackTrace();
      }
   }

   public class CallbackHandler implements InvokerCallbackHandler
   {
      /**
       * Will take the callback message and send back to client.
       * If client locator is null, will store them till client polls to get them.
       *
       * @param invocation
       * @throws org.jboss.remoting.HandleCallbackException
       *
       */
      public void handleCallback(InvocationRequest invocation) throws HandleCallbackException
      {
         if(invocation instanceof Callback)
         {
            Callback callback = (Callback)invocation;
            System.out.println("Received callback value of: " + callback.getCallbackObject());
            System.out.println("Received callback handle object of: " + callback.getCallbackHandleObject());
            System.out.println("Received callback server invoker of: " + callback.getServerLocator());
         }
         else
         {
            System.out.println("Received callback value of: " + invocation.getParameter());
         }
      }
   }

}