/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.oneway;

import org.apache.log4j.Level;
import org.jboss.dtf.MultipleTestRunner;
import org.jboss.logging.Logger;
import org.jboss.remoting.AbstractInvokerTest;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.invocation.NameBasedInvocation;

import java.rmi.server.UID;

/**
 * This is the actual concrete test for the invoker client to test oneway calls (client and server based).
 *
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class OnewayInvokerClientTest extends AbstractInvokerTest
{
   private static final Logger log = Logger.getLogger(OnewayInvokerClientTest.class);

   private String sessionId = new UID().toString();

   private Client client;

   private static final String NAME = "OnewayInvokerClientTest.class";

   public OnewayInvokerClientTest(String name)
   {
      super(NAME);
   }

   public OnewayInvokerClientTest(int numberOfInstances)
   {
      super(NAME, numberOfInstances);
   }

   public OnewayInvokerClientTest(String transport, int port)
   {
      super(NAME, transport, port);
   }

   public OnewayInvokerClientTest(String transport, int port, int numberOfInstances)
   {
      super(NAME, transport, port, numberOfInstances);
   }

   public void init()
   {
      try
      {
         InvokerLocator locator = new InvokerLocator(transport + "://localhost:" + port);
         client = new Client(locator, "test");
         client.connect();
      }
      catch(Exception e)
      {
         log.error(e.getMessage(), e);
      }
   }

   public void runInvokers() throws Throwable
   {
      startup(getNumberOfInstances());
      try
      {
         log.debug("\n\n*****************************" +
                   "Calling testOnewayServerInvocation" +
                   "*******************************\n");
         testOnewayServerInvocation();
         log.debug("\n\n*****************************" +
                   "Calling testOnewayClientInvocation" +
                   "*******************************\n");
         testOnewayClientInvocation();
      }
      finally
      {
         shutdown();
      }
   }

   /**
    * Test simple oneway client invocation
    *
    * @throws Throwable
    */
   public void testOnewayServerInvocation() throws Throwable
   {
      try
      {
         log.debug("running testOnewayClientCallback()");

         sessionId = new UID().toString();
         init();

         sessionId = client.getSessionId();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());

         // simple invoke
         String param = "bar";
         makeServerOnewayInvocation("saveInvocationParameter", param);
         Thread.currentThread().sleep(1000);
         Object obj = makeInvocation("getLastInvocationParameter", null);

         checkAssertion(param, obj);
      }
      finally
      {
         if(client != null)
         {
            client.disconnect();
         }
      }
   }

   protected void checkAssertion(String param, Object obj)
   {
      assertEquals(param, obj);
   }

   protected void makeServerOnewayInvocation(String method, String param) throws Throwable
   {
      client.invokeOneway(new NameBasedInvocation(method,
                                                  new Object[]{param},
                                                  new String[]{String.class.getName()}),
                          null,
                          false);

   }

   /**
    * Test simple oneway client invocation
    *
    * @throws Throwable
    */
   public void testOnewayClientInvocation() throws Throwable
   {
      try
      {
         log.debug("running testOnewayClientCallback()");

         sessionId = new UID().toString();
         init();

         sessionId = client.getSessionId();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());


         // simple invoke
         String param = "bar";
         makeClientOnewayInvocation("saveInvocationParameter", param);
         Thread.currentThread().sleep(1000);
         Object obj = makeInvocation("getLastInvocationParameter", null);

         checkAssertion(param, obj);

      }
      finally
      {
         if(client != null)
         {
            client.disconnect();
         }
      }
   }

   /**
    * Test simple oneway client invocation
    *
    * @throws Throwable
    */
   public void testClientInvocation() throws Throwable
   {
      try
      {
         log.debug("running testInvocation()");

         sessionId = new UID().toString();
         init();

         sessionId = client.getSessionId();

         log.debug("client.getInvoker().getLocator()" + client.getInvoker().getLocator());


         // simple invoke
         String param = "bar";
         Object resp = makeClientInvocation("saveInvocationParameter", param);

         Object obj = makeInvocation("getLastInvocationParameter", null);
         Thread.currentThread().sleep(1000);
         checkAssertion(param, obj);

      }
      finally
      {
         if(client != null)
         {
            client.disconnect();
         }
      }
   }

   protected void makeClientOnewayInvocation(String method, String param) throws Throwable
   {
      client.invokeOneway(new NameBasedInvocation(method,
                                                  new Object[]{param},
                                                  new String[]{String.class.getName()}),
                          null,
                          true);

   }

   protected Object makeClientInvocation(String method, String param) throws Throwable
   {
      Object ret = client.invoke(new NameBasedInvocation(method,
                                                         new Object[]{param},
                                                         new String[]{String.class.getName()}),
                                 null);

      return ret;
   }


   protected Object makeInvocation(String method, String param) throws Throwable
   {
      Object ret = client.invoke(new NameBasedInvocation(method,
                                                         new Object[]{param},
                                                         new String[]{String.class.getName()}),
                                 null);

      return ret;
   }


   public static void main(String[] args)
   {

      org.apache.log4j.BasicConfigurator.configure();
      org.apache.log4j.Category.getRoot().setLevel(Level.INFO);
      org.apache.log4j.Category.getInstance("org.jgroups").setLevel(Level.INFO);
      org.apache.log4j.Category.getInstance("org.jboss.remoting").setLevel(Level.DEBUG);
      org.apache.log4j.Category.getInstance("test").setLevel(Level.DEBUG);

      OnewayInvokerClientTest client = null;
      if(args.length == 1)
      {
         int instances = Integer.parseInt(args[0]);
         client = new OnewayInvokerClientTest(instances);
      }
      else if(args.length == 2)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         client = new OnewayInvokerClientTest(transport, port);
      }
      else if(args.length == 3)
      {
         String transport = args[0];
         int port = Integer.parseInt(args[1]);
         int instances = Integer.parseInt(args[2]);
         client = new OnewayInvokerClientTest(transport, port, instances);
      }
      else
      {
         client = new OnewayInvokerClientTest(OnewayInvokerClientTest.class.getName());
         System.out.println("Using default transport (" + client.getTransport() +
                            ") and default port (" + client.getPort() + ") and " +
                            "default number of instances (" + client.getNumberOfInstances() + ")" +
                            "\nCan enter transport, port, and instances via command line.");
      }

      try
      {
         //regular class run
         //client.runInvokers();
         MultipleTestRunner runner = new MultipleTestRunner();
         runner.doRun(client, true);
      }
      catch(Throwable e)
      {
         e.printStackTrace();
         System.exit(1);
      }
      System.exit(0);
   }
}
