/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.remoting.callback;

import org.jboss.dtf.MultipleTestRunner;
import org.jboss.logging.Logger;
import org.jboss.remoting.AbstractInvokerTest;
import org.jboss.remoting.Callback;
import org.jboss.remoting.Client;
import org.jboss.remoting.HandleCallbackException;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerCallbackHandler;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.transport.Connector;
import org.jboss.samples.callback.CallbackServer;
import org.apache.log4j.Level;

import javax.management.MBeanServer;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

/**
 * @author <a href="mailto:telrod@e2technologies.net">Tom Elrod</a>
 */
public class CallbackTestClient extends AbstractInvokerTest
{
   // Default locator values
   private static String transport = "socket";
   private static String host = "localhost";
   private static int port = 5500;

   private String locatorURI = null;

   private Client remotingClient;

   private static final Logger log = Logger.getLogger(CallbackTestClient.class);

   public static final String RESPONSE_VALUE = "This is the return to SampleInvocationHandler invocation";


   public CallbackTestClient(String name)
   {
      super(name);
   }

   public CallbackTestClient(String name, int numberOfInstances)
   {
      super(name, numberOfInstances);
   }

   public CallbackTestClient(String name, String transport, int port)
   {
      super(name, transport, port);
   }

   public CallbackTestClient(String name, String transport, int port, int numberOfInstances)
   {
      super(name, transport, port, numberOfInstances);
   }

   public CallbackTestClient(String name, String locatorURI)
   {
      super(name);
      this.locatorURI = locatorURI;
   }

   private void init() throws Exception
   {
//      if(remotingClient == null)
//      {
         createRemotingClient();
//      }
   }

   public void createRemotingClient() throws Exception
   {
      InvokerLocator locator = new InvokerLocator(locatorURI);
      System.out.println("Calling remoting server with locator uri of: " + locatorURI);

      // This could have been new Client(locator), but want to show that subsystem param is null
      // Could have also been new Client(locator, "sample");
      remotingClient = new Client(locator, null);

   }

   public void makeInvocation() throws Throwable
   {
      Object response = remotingClient.invoke("Do something", null);
      System.out.println("Invocation response: " + response);
   }

   public void testPullCallback() throws Throwable
   {
      init();

      CallbackHandler callbackHandler = new CallbackHandler();
      // by passing only the callback handler, will indicate pull callbacks
      remotingClient.addListener(callbackHandler);
      // now make invocation on server, which should cause a callback to happen
      makeInvocation();

      List callbacks = remotingClient.getCallbacks();
      Iterator itr = callbacks.iterator();
      while(itr.hasNext())
      {
         Object obj = itr.next();
         log.info("testPullCallback - Callback object should have been " +
                  CallbackTestServer.CALLBACK_VALUE + " and was " +
                  (obj == null ? null : ((Callback)obj).getCallbackObject()));
         if(obj instanceof Callback)
         {
            assertEquals("Callback object is NOT same.", CallbackTestServer.CALLBACK_VALUE, ((Callback) obj).getCallbackObject());
         }
         else
         {
            assertTrue("Callback object is NOT of type Callback.", false);
         }
      }

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);
      remotingClient.disconnect();
   }

   public void testPushCallback() throws Throwable
   {
      init();

      // Using loctor with port value one higher than the target server
      String callbackLocatorURI = transport + "://" + host + ":" + (port + 5);
      InvokerLocator callbackLocator = new InvokerLocator(callbackLocatorURI);

      log.info("testPushCallback - Setting up server.");
      // call to create remoting server to
      // receive client callbacks.
      setupServer(callbackLocator);

      CallbackHandler callbackHandler = new CallbackHandler();
      // Callback handle object will be passed back as part of the callback object
      String callbackHandleObject = "myCallbackHandleObject";
      log.info("testPushCallback - adding listener.");
      // by passing only the callback handler, will indicate pull callbacks
      remotingClient.addListener(callbackHandler, callbackLocator, callbackHandleObject);
      log.info("testPushCallback - make invocation");
      // now make invocation on server, which should cause a callback to happen
      makeInvocation();

      // need to wait for brief moment so server can callback
      Thread.sleep(5000);

      Callback callback = callbackHandler.getCallback();
      log.info("testPushCallback - Callback returned was " + callback);
      assertNotNull("Callback returned was null.", callback);

      Object callbackObj = callback.getCallbackObject();
      log.info("testPushCallback - Callback value should have been " + CallbackTestServer.CALLBACK_VALUE + ", and was " + callbackObj);
      assertEquals("Callback value should have been " + CallbackTestServer.CALLBACK_VALUE + ", but was " + callbackObj,
                   CallbackTestServer.CALLBACK_VALUE, callbackObj);
      Object callbackHandleObj = callback.getCallbackHandleObject();
      log.info("testPushCallback - Callback handle object should have been " + callbackHandleObject + ", and was " + callbackHandleObj);
      assertEquals("Callback handle object should have been " + callbackHandleObject + ", but was " + callbackHandleObj,
                   callbackHandleObject, callbackHandleObj );
      InvokerLocator serverLoc = callback.getServerLocator();
      log.info("testPushCallback - Callback server locator should have been " + remotingClient.getInvoker().getLocator() +
                   ", and was " + serverLoc);
      assertEquals("Callback server locator should have been " + remotingClient.getInvoker().getLocator() +
                   ", but was " + serverLoc,
                   remotingClient.getInvoker().getLocator(), serverLoc);

      // remove callback handler from server
      remotingClient.removeListener(callbackHandler);

   }

   public void setupServer(InvokerLocator locator) throws Exception
   {
      log.info("Starting remoting server with locator uri of: " + locator);
      try
      {
         Connector connector = new Connector();
         connector.setInvokerLocator(locator.getLocatorURI());
         connector.start();

         SampleInvocationHandler invocationHandler = new SampleInvocationHandler();
         // first parameter is sub-system name.  can be any String value.
         connector.addInvocationHandler("sample", invocationHandler);
      }
      catch(Exception e)
      {
         log.error("Error starting callback server", e);
         throw e;
      }
   }


   /**
    * Can pass transport and port to be used as parameters.
    * Valid transports are 'rmi' and 'socket'.
    *
    * @param args
    */
   public static void main(String[] args)
   {
      if(args != null && args.length == 2)
      {
         transport = args[0];
         port = Integer.parseInt(args[1]);
      }
      String locatorURI = transport + "://" + host + ":" + port;
      CallbackTestClient client = new CallbackTestClient(CallbackTestClient.class.getName(), locatorURI);

      //client.setLogging();
      try
      {
         MultipleTestRunner runner = new MultipleTestRunner();
         runner.doRun(client, true);
      }
      catch(Throwable e)
      {
         e.printStackTrace();
         System.exit(1);
      }
      System.exit(0);
   }

   /**
    * Simple invocation handler implementation.
    */
   public static class SampleInvocationHandler implements ServerInvocationHandler
   {

      private List listeners = new ArrayList();


      /**
       * called to handle a specific invocation
       *
       * @param invocation
       * @return
       * @throws Throwable
       */
      public Object invoke(InvocationRequest invocation) throws Throwable
      {
         // Just going to return static string as this is just simple example code.

         // Will also fire callback to listeners if they were to exist using
         // simple invocation request.
         Callback callback = new Callback("This is the payload of callback invocation.");
         Iterator itr = listeners.iterator();
         while(itr.hasNext())
         {
            InvokerCallbackHandler callbackHandler = (InvokerCallbackHandler) itr.next();
            try
            {
               callbackHandler.handleCallback(callback);
            }
            catch(HandleCallbackException e)
            {
               e.printStackTrace();
            }
         }

         return RESPONSE_VALUE;

      }

      /**
       * Adds a callback handler that will listen for callbacks from
       * the server invoker handler.
       *
       * @param callbackHandler
       */
      public void addListener(InvokerCallbackHandler callbackHandler)
      {
         listeners.add(callbackHandler);
      }

      /**
       * Removes the callback handler that was listening for callbacks
       * from the server invoker handler.
       *
       * @param callbackHandler
       */
      public void removeListener(InvokerCallbackHandler callbackHandler)
      {
         listeners.remove(callbackHandler);
      }

      /**
       * set the mbean server that the handler can reference
       *
       * @param server
       */
      public void setMBeanServer(MBeanServer server)
      {
         // NO OP as do not need reference to MBeanServer for this handler
      }

      /**
       * set the invoker that owns this handler
       *
       * @param invoker
       */
      public void setInvoker(ServerInvoker invoker)
      {
         // NO OP as do not need reference back to the server invoker
      }

   }


   public class CallbackHandler implements InvokerCallbackHandler
   {
      private Callback callback;

      /**
       * Will take the callback message and send back to client.
       * If client locator is null, will store them till client polls to get them.
       *
       * @param invocation
       * @throws org.jboss.remoting.HandleCallbackException
       *
       */
      public void handleCallback(InvocationRequest invocation) throws HandleCallbackException
      {
         if(invocation instanceof Callback)
         {
            callback = (Callback) invocation;
            System.out.println("Received callback value of: " + callback.getCallbackObject());
            System.out.println("Received callback handle object of: " + callback.getCallbackHandleObject());
            System.out.println("Received callback server invoker of: " + callback.getServerLocator());
         }
         else
         {
            System.out.println("Received callback value of: " + invocation.getParameter());
         }
      }

      public Callback getCallback()
      {
         return callback;
      }

   }

}