/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.media.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import java.net.URL;

import org.jboss.util.stream.Streams;

/**
 * Resource utilities.
 *
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo Argello</a>
 */
public class ResourceUtils
{
   public static InputStream createStreamFromResource(String resourceName)
   {
      ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
      return classLoader.getResourceAsStream(resourceName);
   }

   public static File createFileFromResource(
      String resourceName,
      String filePrefix,
      String fileSuffix)
      throws IOException
   {
      InputStream inputStream = createStreamFromResource(resourceName);

      try
      {
         File tempFile = File.createTempFile(filePrefix, fileSuffix);
         tempFile.deleteOnExit();
         OutputStream tempFileStream = new FileOutputStream(tempFile);

         try
         {
            Streams.copyb(inputStream, tempFileStream);
         }
         finally
         {
            tempFileStream.close();
         }

         return tempFile;
      }
      finally
      {
         inputStream.close();
      }
   }

   /**
    * Return the content of the resource provided as a byte array.
    *
    * @param resource name.
    * @return content as a byte array.
    */
   public static byte[] loadBinaryData(String resourceName) throws IOException
   {
      InputStream inputStream =
         ResourceUtils.createStreamFromResource(resourceName);

      try
      {
         ByteArrayOutputStream ouputStream = new ByteArrayOutputStream();
         try
         {
            int byteRead;
            while ((byteRead = inputStream.read()) != -1)
               ouputStream.write(byteRead);
            return ouputStream.toByteArray();
         }
         finally
         {
            ouputStream.close();
         }
      }
      finally
      {
         inputStream.close();
      }
   }

   /**
    * Return the content of the resource provided as a String.
    *
    * @param resource name.
    * @return content as a string.
    */
   public static String loadTextData(String resourceName) throws IOException
   {
      InputStream inputStream =
         ResourceUtils.createStreamFromResource(resourceName);

      try
      {
         StringWriter stringWriter = new StringWriter();
         try
         {
            int byteRead;
            while ((byteRead = inputStream.read()) != -1)
               stringWriter.write(byteRead);
            return stringWriter.toString();
         }
         finally
         {
            stringWriter.close();
         }
      }
      finally
      {
         inputStream.close();
      }
   }

   public static InputStream getResourceStream(String resource)
   {
      return Thread
         .currentThread()
         .getContextClassLoader()
         .getResourceAsStream(
         resource);
   }

   public static File getResourceFile(String resource)
   {
      URL u =
         Thread.currentThread().getContextClassLoader().getResource(resource);
      String file = u.toExternalForm();
      return new File(file.substring(6, file.length()));
   }
}
