/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package javax.emb;

import java.io.Serializable;

/**
 * This interface defines the common behavior of media listener. Such observers
 * are implemented by application programmers to be persisted with and extend
 * the semantics of media entity EJBs in a way that does not depend on the
 * implementation of the latter. Listeners are notified of specific events in
 * media entity EJBs once they are registered with said. As they have to be
 * persisted with media entity EJBs, all listeners must be serializable.
 * 
 * <p>The persistent observer design differs from the standard Java listener
 * practice by passing the name of properties affected instead of passing the
 * property value to distinct callback methods. The reason for this is that the
 * large size of media content makes it practically impossible to pass it to
 * observers. Therefore, observers are forced to retrieve property values from
 * entity beans when there is a need. Please note that Entity Beans must be
 * deployed as reentrant in order for the latter to work.
 * 
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo
 *         Argello</a>
 */
public interface MediaListener extends Serializable
{
   /**
    * This method handles the fact that the given media entity is about to
    * change the value of its property named <code>propertyName</code>.
    * Valid property names are "content", "location", "mimeType", "name" and
    * "description".
    * 
    * @param mediaEntity media entity.
    * @param propertyName prppoerty name.
    * @throws javax.emb.ListenerVetoException if the listener does not agree to
    *         the intended change.
    * @throws javax.emb.MediaException if there's an application specific
    *         problem.
    * @throws java.lang.NullPointerException is thrown if one of the values
    *         passed is <code>null</code>.
    */
   void handleAboutToChangeMediaEntity(
      MediaEntityLocal mediaEntity,
      String propertyName)
      throws MediaException;

   /**
    * This method handles the fact that the given media entity is about to be
    * removed.
    * 
    * @param mediaEntity media entity.
    * @throws javax.emb.ListenerVetoException if the listener does not agree to
    *         the intended removal.
    * @throws javax.emb.MediaException if there's an application specific
    *         problem.
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    */
   void handleAboutToRemoveMediaEntity(MediaEntityLocal mediaEntity)
      throws MediaException;

   /**
    * This method handles the fact that the given media entity has changed the
    * value of its property named propertyName. Valid property names are
    * "content", "location", "mimeType", "name" and "description".
    * 
    * @param mediaEntity media entity.
    * @param propertyName property name.
    * @throws javax.emb.ListenerVetoException if the listener does not agree to
    *         the change performed.
    * @throws javax.emb.MediaException if there's an application specific
    *         problem.
    * @throws java.lang.NullPointerException if one of the values passed is
    *         <code>null</code>.
    */
   void handleMediaEntityChanged(
      MediaEntityLocal mediaEntity,
      String propertyName)
      throws MediaException;

   /**
    * This method overwrites the matching method in <code>java.lang.Object</code>.
    * The result is <code>true</code> if the given object is a <code>MediaListener</code>
    * and it is content equal to the receiver, otherwise <code>false</code>.
    * Note that this behavior is required as listeners are stored and loaded as
    * part of EJB operations and therefore frequently change their object
    * identity.
    * 
    * @param object object.
    */
   boolean equals(Object object);

   /**
    * This method overwrites the matching method in <code>java.lang.Object</code>.
    * The resulting hash code is calculated based on the receiver's content.
    * Note that this behavior is required as listeners are stored and loaded as
    * part of EJB operations and therefore frequently change their object
    * identity.
    * 
    * @return hashcode.
    */
   int hashCode();
}