/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package javax.emb;

import java.net.URL;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.EJBLocalHome;
import javax.ejb.FinderException;

/**
 * This interface defines the local home interface of media entity EJBs and
 * offers services to create and to query persistent and mutable media objects.
 * It extends javax.ejb.EJBLocalHome.
 * 
 * <p>The create method provided allows the creation of empty media entity
 * EJBs. These instances need to be filled with content in a subsequent step,
 * by using the setName() and setContent() methods.
 * 
 * <p>Sometimes networks of associated non-embedded media content can become
 * quite complex. In order to allow the consistent handling of such complex
 * content structures the importMedia() and exportMedia() operations allows for
 * consistent import, creation and export or content if for example many
 * non-embedded parents share identical children. Using the related methods in
 * the media entity EJB component interfaces would lead to multiple versions of
 * the shared children being imported/exported.
 * 
 * <p>Finally, the publishContent() and publishMedia() operations publish a
 * transient or persistent media object to a protocol server, and returns a URL
 * or content that allows clients to access the protocol and server specific
 * copies of said medium and its children. This abstraction allows applications
 * to access proprietary content delivery services, without exposing the
 * application to the details of the implementation of those services. For
 * example, multiple and distributed streaming server technologies can be used
 * underneath a publish implementation. Implementations of this specification
 * are not bound to a specific algorithm to determine a suitable protocol
 * server. Theoretical possibilities range from dynamic selection from a
 * pre-configured server pool to relating suitable protocol servers statically
 * with each media entity. The resulting URL/content is either pointing
 * directly to the cached copy of the media object with a protocol type of
 * http, rtsp or similar, or it is an http URL pointing to a managed
 * meta-medium that points to said cached copy.
 * 
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo
 *         Argello</a>
 */
public interface MediaEntityLocalHome extends EJBLocalHome
{
   final static byte TRANSFER_TYPE_STREAM = 0;
   final static byte TRANSFER_TYPE_BURST = 1;

   /**
    * Creates a new media entity EJB with a generated identity key, the value
    * of <code>System.currentTimeMillis()</code> as value of the <code>lastModified</code>
    * property, and <code>null</code> as value for all other properties.
    * Please note most of the methods of the Media Entity returned will throw
    * exceptions until the name and either the location or the content
    * properties are set.
    * 
    * @return media entity bean.
    * @throws javax.ejb.CreateException is thrown if a problem related to EJB
    *         creation occurs.
    * @throws MediaException
    */
   MediaEntityLocal create() throws CreateException, MediaException;

   /**
    * Copies the content of the given media entity EJBs and all their direct
    * and indirect children to the given targetDirectoryLocation, i.e. a URL of
    * protocol type "file" or any other type supported by the implementation.
    * The operation returns an array of URLs of the same size as the source
    * media array. The elements of the resulting array point to file copies of
    * the source media array elements' content. If the media objects affected
    * are non-embedded, the content of the copies is modified to correctly
    * point to the copies of the children's content instead of the original
    * locations.
    * 
    * <p>If the given <code>targetDirectoryLocation</code> is <code>null</code>,
    * a default export directory location is used. The operation makes sure
    * that each media entity EJB affected corresponds to exactly one media
    * object created. Note that the names of the copied media objects may
    * differ from the original ones, that the operation makes sure no existing
    * files are overwritten by this operation, and that implementations are
    * permitted to copy children to subdirectories of the given
    * targetDirectoryLocation. Also note that the resulting <code>targetLocation</code>
    * URLs are returned in the order of their respective media entity EJBs.
    * Duplicate media entity EJBs result in <code>targetLocation</code> URLs
    * returned in multiple array slots, <code>null</code> media entity EJBs
    * result in respective <code>null</code> values in the result. Finally,
    * note that not all media objects exported are necessarily returned, just
    * the ones for which corresponding media entity EJBs were passed.
    * 
    * @param sourceMedia
    * @param targetDirectoryLocation
    * @return URLs array
    * @throws java.lang.NullPointerException if the given source media array is
    *         <code>null</code>.
    * @throws javax.emb.ContentAccessException in case content transfer fails.
    * @throws javax.emb.MediaFormatException if there is a problem dealing with
    *         one of the media formats involved.
    * @throws javax.emb.MalformedLocationException if the URL given is
    *         malformed or has a protocol type not supported by the
    *         implementation.
    */
   URL[] exportMedia(
      MediaEntityLocal[] sourceMedia,
      URL targetDirectoryLocation)
      throws MediaException;

   /**
    * Returns all media entities in the persistent store whose description
    * contains the given String. The query is case-sensitive. Note that passing
    * an empty string will return all media entity beans in the persistent
    * store that have a description different from <code>null</code>.
    * Passing in <code>null</code> or a partial description not present in
    * the data store will result in an empty collection.
    * 
    * @param partialDescription description.
    * @return all media entities whose description contains the given String.
    * @throws javax.ejb.FinderException if a finder problem occurs.
    */
   Collection findByPartialDescription(String partialDescription)
      throws FinderException;

   /**
    * Returns all media entities in the persistent store whose location URL
    * contains the given String. The query is case-sensitive. Note that passing
    * an empty string will return all media entity beans in the persistent
    * store that have a location different from <code>null</code>. Passing
    * in <code>null</code> or a partial location not present in the data
    * store will result in an empty collection.
    * 
    * @param partialLocation location.
    * @return all media entities in the persistent store whose location URL
    *         contains the given String.
    * @throws javax.ejb.FinderException if a finder problem occurs.
    */
   Collection findByPartialLocation(String partialLocation)
      throws FinderException;

   /**
    * Returns the media entity with the given identity.
    * 
    * @param identity identity.
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    * @throws javax.ejb.FinderException if a finder problem occurs or no media
    *         entity is found.
    */
   MediaEntityLocal findByPrimaryKey(String identity) throws FinderException;

   /**
    * Imports the given media objects and all their direct and indirect
    * children from the given source locations and creates new EJBs for said
    * media objects and children. For the media entity EJBs directly created
    * from the source locations given, the names are derived from the given
    * name array at equivalent indices. If a name value is null a generated
    * name is used, possibly based on the related source location. For all
    * other media entity EJBs created, the name properties are derived from the
    * links present in their respective parent content.
    * 
    * <p>The operation returns an array of media entity EJBs of the same size
    * as the source location array. The elements of the resulting array are
    * based on the content of the corresponding elements of the source location
    * array.
    * 
    * <p>The operation makes sure that each media object affected corresponds
    * to exactly one media entity EJB created in the operations context. Note
    * that the resulting Media ent ity EJBs are returned in the order of their
    * respective sourceLocations. Duplicate sourceLocations result in media
    * entity EJBs returned in multiple array slots, null sourceLocations result
    * in respective null values in the result. If a source location is given
    * multiple times with differing names, one of the names is picked for the
    * resulting media entity EJB. Finally, note that not all media entity EJBs
    * created are necessarily returned, just the ones for which corresponding
    * location URLs were passed.
    * 
    * @param sourceLocations source locations.
    * @param names names.
    * @return @throws java.lang.NullPointerException if one of the arrays
    *         passed is <code>null</code>.
    * @throws java.lang.IllegalArgumentException if the two arrays passed
    *         mismatch in length or if one or more of the name arguments passed
    *         contain invalid characters.
    * @throws javax.ejb.CreateException if the creation of a media entity EJB
    *         fails.
    * @throws javax.emb.ContentAccessException if content transfer fails.
    * @throws javax.emb.MediaFormatException if there's a problem dealing with
    *         one of the media formats involved.
    * @throws javax.emb.MalformedLocationException if one of the URLs given is
    *         malformed or has a protocol type not supported by the
    *         implementation.
    * @throws javax.emb.ContentTooLargeException if one the media entity EJB
    *         contents would become larger than supported by the
    *         implementation.
    */
   MediaEntityLocal[] importMedia(URL[] sourceLocations, String[] names)
      throws CreateException, MediaException;

   /**
    * 
    * This operation publishes persistent or transient media content to a
    * protocol server and returns a URL. The URL allows web clients to access
    * the protocol and server specific copies of said medium. Note that it also
    * exports its children if a non-embedded media entity EJB is passed. If the
    * medium is persistent (i.e. is a media entity EJB), said copies will
    * likely be managed and updated once the original content changes.
    * Optionally, the selection of a suitable protocol server can be influenced
    * using the given constraints object. Implementations of this specification
    * are not bound to a specific algorithm to determine a suitable protocol
    * server. Theoretical possibilities range from dynamic selection from a
    * pre-configured server pool to relating suitable protocol servers
    * statically with each media entity. The resulting URL is either pointing
    * directly to the cached copy of the media object with a protocol type of
    * http, rtsp or similar, or it is an http URL pointing to a managed
    * meta-medium that points to said cached copy.
    * 
    * <p>The kind of protocol server chosen may be determined by the given
    * transferType. Specified values are:
    * 
    * <ul><li>TRANSFER_TYPE_STREAM for streaming of media content, using a
    * stream server as protocol server.</li>
    * 
    * <li>TRANSFER_TYPE_BURST for burst transfer of media content, usually
    * using a web server as protocol server. <li></ul>
    * 
    * <p>Specified protocol constraint types are:
    * 
    * <ul><li>"SERVER_TYPE" for restriction of stream servers to chose from.
    * The constraint value is an array of Strings. Specified constraint values
    * are "VideoCharger" for IBM VideoCharger servers, "RealSystem" for
    * RealNetworks RealSystem servers, and "Windows Media" for Microsoft
    * Windows Media services, and "QuickTime" for Apple QuickTime servers. The
    * protocol server selected will be of one of these types. Note that some
    * server types can feed a number of different clients.</li>
    * 
    * <li>"CLIENT_TYPE" for restriction of stream servers to chose from. The
    * constraint value is an array of Strings. Specified constraint values are
    * "VideoCharger" for IBM VideoCharger players, "RealPlayer" for
    * RealNetworks players, and "Windows Media" for Microsoft Windows Media
    * players, and "QuickTime" for Apple QuickTime players. The protocol server
    * selected will support one of these client types. Note that some players
    * can work with a number of protocol servers.</li></ul>
    * 
    * <p>In case the selected protocol server supports more than one client
    * type, the client type is not constrained and a meta-medium has to be
    * generated, said meta-medium is generated for the default client type of
    * the selected protocol server. If in a similar case there is more than one
    * constraint value for the client type, the constraint value is interpreted
    * as a priority list and therefore the meta-medium is generated for the
    * first client type in that list that is supported by the selected protocol
    * server.
    * 
    * @param content content.
    * @param transferType transfer type.
    * @param constraints constraints.
    * @return URL
    * @throws java.lang.NullPointerException if the media object passed is
    *         <code>null</code>.
    * @throws java.lang.IllegalArgumentException if a transfer type is passed
    *         that is not supported by the implementation, or if a protocol
    *         type <code>SERVER_TYPE</code> or <code>CLIENT_TYPE</code> is
    *         passed with a value that is not an array of Strings.
    * @throws javax.emb.NoServerFoundException if no suitable protocol server
    *         can be determined.
    * @throws javax.emb.ContentAccessException if content transfer fails or if
    *         non-embedded content is passed in anything other than a media
    *         entity EJB.
    * @throws javax.emb.MediaFormatException if there's a problem dealing with
    *         one of the media formats involved.
    * @throws javax.emb.LinkTranslationException if ingesting protocol specific
    *         links into copies of non-embedded media objects fails.
    * @throws javax.emb.ConversionException if the middleware fails converting
    *         the media content to another format in order to make it suitable
    *         for a specific protocol server.
    */
   URL publishContent(
      Media content,
      byte transferType,
      ProtocolConstraints constraints)
      throws MediaException;

   /**
    * This operation publishes the given playlist of persistent media entities
    * to a protocol server and returns a meta medium that allows clients to
    * access the protocol and server specific copies of said playlist and its
    * children. If the implementation chooses to cache the media objects on the
    * protocol server chosen, it must be guaranteed that eventual child copies
    * are also managed and will be updated if the original content changes.
    * 
    * <p>Optionally, the selection of a suitable protocol server can be
    * influenced using the given constraints object. Implementations of this
    * specification are not bound to a specific algorithm to determine a
    * suitable protocol server. Theoretical possibilities range from dynamic
    * selection from a pre-configured server pool to relating suitable protocol
    * servers statically with each media entity. The content of the resulting
    * meta medium is pointing directly to the cached copies of the given media
    * entities with a protocol type of http, rtsp or similar.
    * 
    * <p>The kind of protocol server chosen may be determined by the given
    * transferType. Specified values are:
    * 
    * TRANSFER_TYPE_STREAM</code> for streaming of media content, using a
    * stream server as protocol server.</li>
    * 
    * <li><code>TRANSFER_TYPE_BURST</code> for burst transfer of media
    * content, usually using a web server as protocol server.</li></ul>
    * 
    * <p>Specified protocol constraint types are:
    * 
    * <ul><li>"SERVER_TYPE" for restriction of stream servers to chose from.
    * The constraint value is an array of Strings. Specified constraint values
    * are "VideoCharger" for IBM VideoCharger servers, "RealSystem" for
    * RealNetworks servers, and "Windows Media" for Microsoft Media services,
    * and "QuickTime" for Apple QuickTime servers. The protocol server selected
    * will be of one of these types. Note that some server types can feed a
    * number of different clients.</li>
    * 
    * <li>"CLIENT_TYPE" for restriction of stream servers to chose from. The
    * constraint value is an array of Strings. Specified constraint values are
    * "VideoCharger" for IBM VideoCharger players, "RealPlayer" for
    * RealNetworks players, and "Windows Media" for Microsoft Windows Media
    * players, and "QuickTime" for Apple QuickTime players. The protocol server
    * selected will support one of these client types. Note that some players
    * can work with a number of protocol servers.</li></ul>
    * 
    * <p>If the selected protocol server supports more than one client type
    * and the client type is not constrained, a meta-medium is generated for
    * the default client type of the selected protocol server. If in a similar
    * case there is more than one constraint value for the client type, the
    * constraint value is interpreted as a priority list and therefore the
    * meta-medium is generated for the first client type in that list that is
    * supported by the selected protocol server.
    * 
    * <p>Both the format and the content of the resulting meta-medium are
    * considered proprietary to an implementation of this specification.
    * Suitable are streaming metafile formats like Real .ram or IBM .ivs for
    * streaming, or HTML files - containing auto load links for playlists
    * consisting of exactly one entry, or clickable links for playlists
    * consisting of two or more entries - for burst transfer. However, a
    * Servlet must be able to return the resulting meta-medium directly to its
    * requesting client to cause a media player to render the playlist on said
    * client.
    * 
    * <p>In contrast, publishing the same playlist for the HTML protocol would
    * have resulted in an html meta media containing HREF links to the SMIL
    * presentation and the MPEG movie. Also, the cache copy of the SMIL
    * presentation would contain HTML references to its child media in this
    * case.
    * 
    * @param playlist
    * @param transferType
    * @param constraints
    * @return @throws java.lang.NullPointerException if the playlist passed is
    *         <code>null</code>.
    * @throws java.lang.IllegalArgumentException if a transfer type is passed
    *         that is not supported by the implementation, or if a protocol
    *         type <code>SERVER_TYPE</code> or <code>CLIENT_TYPE</code> is
    *         passed with a value that is not an array of Strings.
    * @throws javax.emb.NoServerFoundException if no suitable protocol server
    *         can be determined.
    * @throws javax.emb.ContentAccessException if content transfer fails.
    * @throws javax.emb.MediaFormatException if there's a problem dealing with
    *         one of the media formats involved.
    * @throws javax.emb.LinkTranslationException if ingesting protocol specific
    *         links into copies of non-embedded media objects fails.
    * @throws javax.emb.ConversionException if the middleware fails converting
    *         the media content to another format in order to make it suitable
    *         for a specific protocol server.
    */
   Media publishMedia(
      MediaEntityLocal[] playlist,
      byte transferType,
      ProtocolConstraints constraints)
      throws MediaException;
}