/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */

package javax.emb;

import java.io.InputStream;
import java.io.OutputStream;

/**
 * A media converter performs actual conversions of media content.
 * Implementations of this interface can work in one of two ways: Synchronous
 * converters perform conversions completely before returning the input stream
 * on the converted content. Asynchronous converters return a piped input
 * stream immediately after checking the input and spawning a new thread to
 * perform the conversion. The latter has the advantage of being very memory
 * efficient in case the result is consumed immediately, while the first tends
 * to be more robust because it doesn't require the maintenance of additional
 * threads of execution.
 * 
 * <p>The design assumes a converter is instantiated for a specific kind of
 * conversion. Therefore, classes implementing this interface should take a
 * media converter spec as a constructor argument. Said media converter spec
 * should contain the necessary conversion parameters, for example the desired
 * bandwidth for a WAV to MP3 converter.
 * 
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:ricardoarguello@users.sourceforge.net">Ricardo
 *         Argello</a>
 */
public interface MediaConverter
{
   /**
    * Converts the media content offered on the given input stream and returns
    * the converted media content as an input stream. Implementations of this
    * method may, but are not required to, spawn a new thread to perform the
    * actual conversion, and return a PipedInputStream immediately.
    * 
    * @param inputStream the input stream to process.
    * @return the processed input stream.
    * @throws java.lang.NullPointerException if the value passed is <code>null</code>.
    * @throws javax.emb.FormatSyntaxException if the content provided with the
    *         input stream does not meet the expected format syntax.
    * @throws javax.emb.FormatFeatureException if the content provided in the
    *         input stream utilizes a format feature not supported by the
    *         receiver.
    * @throws javax.emb.LinkTranslationException if the content provided
    *         contains relative links to child media objects.
    * @throws javax.emb.ContentAccessException if an I/O problem occurs or if
    *         said child links cannot be resolved to child media content.
    * @throws javax.emb.ConversionException if the conversion fails because a
    *         conversion specific problem occurred.
    */
   InputStream process(InputStream inputStream) throws MediaException;

   /**
    * Converts the media content offered on the given input stream and writes
    * the converted media content on the given output stream. In any case, this
    * method will block until the conversion is completed.
    * 
    * @param inputStream the stream to process.
    * @param outputStream the processed stream.
    * @throws java.lang.NullPointerException if one the values passed is <code>null</code>.
    * @throws javax.emb.FormatSyntaxException if the content provided with the
    *         input stream does not meet the expected format syntax.
    * @throws javax.emb.FormatFeatureException if the content provided in the
    *         input stream utilizes a format feature not supported by the
    *         receiver.
    * @throws javax.emb.LinkTranslationException if the content provided
    *         contains relative links to child media objects.
    * @throws javax.emb.ContentAccessException if an I/O problem occurs or if
    *         said child links cannot be resolved to child media content.
    * @throws javax.emb.ConversionException is thrown if the conversion fails
    *         because a conversion specific problem occurred.
    */
   public void process(InputStream inputStream, OutputStream outputStream)
      throws MediaException;
}