/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.axis.message;

// $Id:NameImpl.java,v 1.0, 2005-06-24 19:12:46Z, Robert Worsnop$

import javax.xml.namespace.QName;
import javax.xml.soap.Name;

/**
 * A representation of an XML name. This interface provides methods for getting the local and namespace-qualified names
 * and also for getting the prefix associated with the namespace for the name. It is also possible to get the URI
 * of the namespace.
 *
 * The following is an example of a namespace declaration in an element.
 *
 *    &lt;wombat:GetLastTradePrice xmlns:wombat="http://www.wombat.org/trader">
 *
 * The following shows what the methods in the Name interface will return.
 *
 *    getQualifiedName will return "prefix:LocalName" = "wombat:GetLastTradePrice"
 *    getURI will return "http://www.wombat.org/trader"
 *    getLocalName will return "GetLastTracePrice"
 *    getPrefix will return "wombat"
 *
 * XML namespaces are used to disambiguate SOAP identifiers from application-specific identifiers.
 *
 * Name objects are created using the method SOAPEnvelope.createName, which has two versions.
 * One method creates Name objects with a local name, a namespace prefix, and a namespace URI.
 * The second creates Name objects with just a local name.
 *
 * The following line of code, in which se is a SOAPEnvelope object, creates a new Name object with all three.
 *
 *    Name name = se.createName("GetLastTradePrice", "wombat", "http://www.wombat.org/trader");
 *
 * The following line of code gives an example of how a Name object can be used.
 * The variable element is a SOAPElement object. This code creates a new SOAPElement object with the given name and
 * adds it to element.
 *
 *    element.addChildElement(name);
 *
 * @author Thomas.Diesler (thomas.diesler@jboss.org)
 * @since 01-June-2004
 */
public class NameImpl implements Name
{

   private QName qname;
   private String prefix;

   public NameImpl(String local)
   {
      this(local, null, null);
   }

   public NameImpl(String local, String prefix, String uri)
   {
      qname = new QName(uri, local);
      this.prefix = (prefix == null ? "" : prefix);
   }

   /**
    * Gets the local name part of the XML name that this Name object represents.
    *
    * @return a string giving the local name
    */
   public String getLocalName()
   {
      return qname.getLocalPart();
   }

   /**
    * Returns the prefix that was specified when this Name object was initialized.
    * This prefix is associated with the namespace for the XML name that this Name object represents.
    *
    * @return the prefix as a string
    */
   public String getPrefix()
   {
      return prefix;
   }

   /**
    * Gets the namespace-qualified name of the XML name that this Name object represents.
    *
    * @return the namespace-qualified name as a string
    */
   public String getQualifiedName()
   {
      return prefix + ":" + qname.getLocalPart();
   }

   /**
    * Returns the URI of the namespace for the XML name that this Name object represents.
    *
    * @return the URI as a string
    */
   public String getURI()
   {
      return qname.getNamespaceURI();
   }

   public int hashCode()
   {
      return qname.hashCode();
   }

   /**
    * Equality compares localPart and namespaceURI.
    * The prefix is not relevant for equality.
    */
   public boolean equals(Object obj)
   {
      if (!(obj instanceof NameImpl)) return false;
      if (obj == this) return true;
      NameImpl other = (NameImpl)obj;
      return qname.equals(other.qname);
   }

   public String toString()
   {
      return qname.toString();
   }
}
