/*
* The Apache Software License, Version 1.1
*
*
* Copyright (c) 2001-2003 The Apache Software Foundation.  All rights
* reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* 1. Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
* 2. Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in
*    the documentation and/or other materials provided with the
*    distribution.
*
* 3. The end-user documentation included with the redistribution,
*    if any, must include the following acknowledgment:
*       "This product includes software developed by the
*        Apache Software Foundation (http://www.apache.org/)."
*    Alternately, this acknowledgment may appear in the software itself,
*    if and wherever such third-party acknowledgments normally appear.
*
* 4. The names "Axis" and "Apache Software Foundation" must
*    not be used to endorse or promote products derived from this
*    software without prior written permission. For written
*    permission, please contact apache@apache.org.
*
* 5. Products derived from this software may not be called "Apache",
*    nor may "Apache" appear in their name, without prior written
*    permission of the Apache Software Foundation.
*
* THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
* WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
* ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
* USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
* OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
* SUCH DAMAGE.
* ====================================================================
*
* This software consists of voluntary contributions made by many
* individuals on behalf of the Apache Software Foundation.  For more
* information on the Apache Software Foundation, please see
* <http://www.apache.org/>.
*/

package org.jboss.axis.client;

import org.jboss.axis.AxisFault;
import org.jboss.axis.Constants;
import org.jboss.axis.description.OperationDesc;
import org.jboss.axis.description.ParameterDesc;
import org.jboss.axis.enums.Style;
import org.jboss.axis.enums.Use;
import org.jboss.axis.utils.JavaUtils;
import org.jboss.logging.Logger;

import javax.activation.DataHandler;
import javax.xml.namespace.QName;
import javax.xml.rpc.ServiceException;
import javax.xml.rpc.holders.Holder;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Map;

/**
 * Very simple dynamic proxy InvocationHandler class.  This class is
 * constructed with a Call object, and then each time a method is invoked
 * on a dynamic proxy using this invocation handler, we simply turn it into
 * a SOAP request.
 *
 * @author Glen Daniels (gdaniels@macromedia.com)
 * @author Cdric Chabanois (cchabanois@ifrance.com)
 */
public class AxisClientProxy implements InvocationHandler
{

   private static Logger log = Logger.getLogger(AxisClientProxy.class.getName());

   private Call call;
   private QName portName;

   /**
    * Constructor - package access only (should only really get used
    * in Service.getPort(endpoint, proxyClass).
    * Call can be pre-filled from wsdl
    */
   AxisClientProxy(Call call, QName portName)
   {
      this.call = call;
      this.portName = portName; // can be null
   }


   /**
    * Parameters for invoke method are not the same as parameter for Call
    * instance :
    * - Holders must be converted to their mapped java types
    * - only in and inout parameters must be present in call parameters
    *
    * @param proxyParams proxyParameters
    * @return Object[]   Call parameters
    * @throws JavaUtils.HolderException
    */
   private Object[] proxyParams2CallParams(Object[] proxyParams)
           throws JavaUtils.HolderException, ServiceException
   {
      ArrayList callParams = new ArrayList();

      OperationDesc operationDesc = call.getOperation();
      if (operationDesc == null || proxyParams == null)
      {
         // we don't know which parameters are IN, OUT or INOUT
         // let's suppose they are all in
         return proxyParams;
      }

      // A document/literal method invocation may have more than one parameter,
      // that must be mapped to the single IN parameter wrapper.
      boolean isDocumentLiteral = Style.DOCUMENT.equals(operationDesc.getStyle()) && Use.LITERAL.equals(operationDesc.getUse());
      if (isDocumentLiteral)
      {
         int proxyIndex = 0;
         ArrayList opParams = operationDesc.getParameters();
         for (int i = 0; i < opParams.size(); i++)
         {
            ParameterDesc parameterDesc = (ParameterDesc)opParams.get(i);
            Class javaType = parameterDesc.getJavaType();

            // The javaType is not defined or the proxy param is null
            if (javaType == null || proxyParams[proxyIndex] == null)
            {
               callParams.add(proxyParams[proxyIndex++]);
               continue;
            }

            // Check if the call param type is assignable from the proxy param
            Class proxyType = proxyParams[proxyIndex].getClass();
            if (javaType.isAssignableFrom(proxyType))
            {
               callParams.add(proxyParams[proxyIndex++]);
               continue;
            }

            // Check if the call param type is convertable from the proxy param
            if (JavaUtils.isConvertable(proxyType, javaType))
            {
               Object param = JavaUtils.convert(proxyParams[proxyIndex++], javaType);
               callParams.add(param);
               continue;
            }

            // Get the constructors for the call param type
            boolean ctorFound = false;
            Constructor[] ctors = javaType.getConstructors();
            for (int j = 0; j < ctors.length; j++)
            {
               boolean paramsMatch = true;

               // check if the proxy params match
               Constructor ctor = ctors[j];
               Class[] ctorTypes = ctor.getParameterTypes();
               if (ctorTypes.length > 0)
               {
                  Object[] ctorVals = new Object[ctorTypes.length];
                  for (int k = 0; paramsMatch && k < ctorTypes.length; k++)
                  {
                     if (ctorTypes[k].isAssignableFrom(proxyParams[proxyIndex + k].getClass()))
                        ctorVals[k] = proxyParams[proxyIndex + k];
                     else
                        paramsMatch = false;
                  }

                  if (paramsMatch)
                  {
                     try
                     {
                        Object inst = ctor.newInstance(ctorVals);
                        proxyIndex += ctorVals.length;
                        callParams.add(inst);
                        ctorFound = true;
                        break;
                     }
                     catch (Exception e)
                     {
                        throw new ServiceException("Cannot map proxy params to: " + javaType);
                     }
                  }
               }
            }

            // Let's somebody else sort it out
            if (ctorFound == false)
               callParams.add(proxyParams[proxyIndex++]);
         }
      }

      // not document/literal
      else
      {
         for (int i = 0; proxyParams != null && i < proxyParams.length; i++)
         {
            Object param = proxyParams[i];
            ParameterDesc paramDesc = operationDesc.getParameter(i);
            if (paramDesc == null)
               throw new ServiceException("Cannot obtain parameter " + i + " for: " + operationDesc);

            if (paramDesc.getMode() == ParameterDesc.INOUT)
            {
               callParams.add(JavaUtils.getHolderValue((Holder)param));
            }
            else if (paramDesc.getMode() == ParameterDesc.IN)
            {
               callParams.add(param);
            }
         }
      }

      return callParams.toArray();
   }

   /**
    * Set the return type class on the operation desc
    * TDI 23-June-2004
    */
   private void proxyReturn2CallReturn(Class proxyReturn)
   {

      OperationDesc operationDesc = call.getOperation();
      if (operationDesc != null)
      {
         Class operationReturn = operationDesc.getReturnClass();
         if (proxyReturn != null && operationReturn != null)
         {
            // Always use the DataHandler deserializer if the proxy return
            // type is DataHandler
            if (proxyReturn.equals(DataHandler.class))
            {
               operationDesc.setReturnClass(DataHandler.class);
               operationDesc.setReturnType(Constants.MIME_DATA_HANDLER);
            }

            boolean isConvertible = JavaUtils.isConvertable(operationReturn, proxyReturn);
            if (isConvertible == false)
               isConvertible = getDocLitResultWrapper(operationReturn, proxyReturn) != null;

            if (isConvertible == false)
            {
               log.debug("Fixing return class: " + operationReturn + " -> " + proxyReturn);
               operationDesc.setReturnClass(proxyReturn);
            }
         }

         if (proxyReturn == null && operationReturn != null)
         {
            log.debug("Forcing return class to null: " + operationReturn);
            operationDesc.setReturnClass(null);
         }
      }
   }

   /** With doc/literal it is possible that the call returns a wrapper object that contains the proxy return
    * [TDI] 05-Oct-2004
    */
   private Method getDocLitResultWrapper(Class callReturn, Class proxyReturn)
   {
      Method getter = null;
      OperationDesc operationDesc = call.getOperation();

      // With document/literal, check if operation return has a getter that returns the proxy return
      boolean isDocumentLiteral = Style.DOCUMENT.equals(operationDesc.getStyle()) && Use.LITERAL.equals(operationDesc.getUse());
      if (isDocumentLiteral)
      {
         Method[] methods = callReturn.getMethods();
         for (int i = 0; getter == null && i < methods.length; i++)
         {
            Method method = methods[i];
            if (method.getName().startsWith("get") && method.getParameterTypes().length == 0 && proxyReturn.isAssignableFrom(method.getReturnType()))
            {
               log.debug("Trying to unwrap proxy return with: " + method);
               getter = method;
            }
         }
      }

      return getter;
   }

   /**
    * copy in/out and out parameters (Holder parameters) back to proxyParams
    *
    * @param proxyParams proxyParameters
    */
   private void callOutputParams2proxyParams(Object[] proxyParams)
           throws JavaUtils.HolderException
   {
      OperationDesc operationDesc = call.getOperation();
      if (operationDesc == null || proxyParams == null)
      {
         // we don't know which parameters are IN, OUT or INOUT
         // let's suppose they are all in
         return;
      }

      Map outputParams = call.getOutputParams();

      for (int i = 0; i < operationDesc.getNumParams(); i++)
      {
         Object param = proxyParams[i];
         ParameterDesc paramDesc = operationDesc.getParameter(i);
         if (paramDesc.getMode() != ParameterDesc.IN)
         {
            Object value = outputParams.get(paramDesc.getQName());

            // [TDI 14-Aug-2004] The output param is already a holder
            if (Holder.class.isAssignableFrom(value.getClass()))
               value = JavaUtils.getHolderValue(value);

            JavaUtils.setHolderValue((Holder)param, value);
         }
      }
   }


   /**
    * Handle a method invocation.
    */
   public Object invoke(Object o, Method method, Object[] objects)
           throws Throwable
   {
      try
      {
         if (method.getName().equals("_setProperty"))
         {
            call.setProperty((String)objects[0], objects[1]);
            return null;
         }
         else if (method.getName().equals("_getProperty"))
         {
            return call.getProperty((String)objects[0]);

         }
         else if (method.getName().equals("hashCode"))
         {
            return new Integer(call.hashCode());

         }
         else if (method.getName().equals("toString"))
         {
            return call.toString();

         }
         else if (method.getName().equals("equals"))
         {
            return new Boolean(o == objects[0]);

         }
         else
         {
            Object outValue;
            Object[] paramsCall;

            if ((call.getTargetEndpointAddress() != null) && (call.getPortName() != null))
            {
               // call object has been prefilled : targetEndPoint and portname
               // are already set. We complete it with method informations
               call.setOperation(method.getName());
               paramsCall = proxyParams2CallParams(objects);
               proxyReturn2CallReturn(method.getReturnType());
               outValue = call.invoke(paramsCall);
            }
            else if (portName != null)
            {
               // we only know the portName. Try to complete this information
               // from wsdl if available
               call.setOperation(portName, method.getName());
               paramsCall = proxyParams2CallParams(objects);
               proxyReturn2CallReturn(method.getReturnType());
               outValue = call.invoke(paramsCall);
            }
            else
            {
               // we don't even know the portName (we don't have wsdl)
               paramsCall = objects;
               proxyReturn2CallReturn(method.getReturnType());
               outValue = call.invoke(method.getName(), paramsCall);
            }

            callOutputParams2proxyParams(objects);
            outValue = callReturn2ProxyReturn(outValue, method.getReturnType());

            return outValue;
         }
      }
      catch (AxisFault af)
      {
         if (af.detail != null)
         {
            throw af.detail;
         }
         throw af;
      }
   }

   /** Convert the call return value to the actual proxy return type
    * [TDI] 05-Oct-2004
    */
   private Object callReturn2ProxyReturn(Object callReturn, Class proxyReturnType)
   {
      // Nothing to do
      if (callReturn == null)
         return callReturn;

      // try normal convertion
      if (JavaUtils.isConvertable(callReturn, proxyReturnType))
         return JavaUtils.convert(callReturn, proxyReturnType);

      // convert an ArrayList
      if (callReturn instanceof ArrayList)
         return convertArrayList(callReturn);

      Class callReturnType = callReturn.getClass();
      Method docLitResultWrapper = getDocLitResultWrapper(callReturnType, proxyReturnType);
      if (docLitResultWrapper != null)
      {
         try
         {
            Object proxyReturn = docLitResultWrapper.invoke(callReturn, new Object[]{});
            return proxyReturn;
         }
         catch (Exception e)
         {
            log.error("Cannot unwrap call return", e);
         }
      }

      log.warn("Cannot convert call return " + callReturnType.getName() + " to: " + proxyReturnType.getName());
      return callReturn;
   }

   private Object convertArrayList(Object outValue)
   {
      Object value = ((ArrayList)outValue).toArray();

      //This is a hack due to the array type of unknown types being Object[]
      if (value.getClass().isArray())
      {
         if (!value.getClass().getComponentType().isPrimitive())
         {
            int len = java.lang.reflect.Array.getLength(value);
            Class type = null;
            for (int x = 0; x < len; x++)
            {
               Object o = java.lang.reflect.Array.get(value, x);
               if (o != null)
               {
                  if (type == null)
                  {
                     type = o.getClass();
                  }
                  else
                  {
                     if (!type.getName().equals(o.getClass().getName()))
                     {
                        type = null;
                        break;
                     }
                  }
               }
            }
            // did we find that all elements were of same type
            if (type != null)
            {
               Object convertedArray = java.lang.reflect.Array.newInstance(type, len);
               System.arraycopy(value, 0, convertedArray, 0, len);
               value = convertedArray;
            }
         }
      }

      return value;
   }

   /**
    * Returns the current call
    *
    * @return call
    */
   public Call getCall()
   {
      return call;
   }
}
