/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.axis.attachments;

// $Id:AttachmentSupport.java,v 1.0, 2005-06-24 18:46:12Z, Robert Worsnop$

import org.jboss.axis.AxisEngine;
import org.jboss.axis.MessageContext;
import org.jboss.axis.utils.ClassUtils;
import org.jboss.axis.utils.Messages;
import org.jboss.logging.Logger;

/**
 * Provide access to the attechment support implementation.
 * <p/>
 * The default implementation is {@link AttachmentsImpl}
 * You can overwrite the implementation through the engine option {@link AxisEngine.PROP_ATTACHMENT_IMPLEMENTATION}
 *
 * @author Thomas Diesler (thomas.diesler@jboss.org)
 * @since 31-May-2004
 */
public final class AttachmentSupport
{

   /**
    * Provides logging
    */
   private static Logger log = Logger.getLogger(AttachmentSupport.class.getName());

   /**
    * Default Attachments Implementation class.
    */
   public static final String DEFAULT_ATTACHMNET_IMPL = AxisEngine.DEFAULT_ATTACHMENT_IMPL;

   /**
    * Current Attachment implementation.
    */
   private static String attachImplClassName = DEFAULT_ATTACHMNET_IMPL;

   private static boolean checkForAttachmentSupport = true;
   private static boolean attachmentSupportEnabled = false;
   private static Class attachImplClass;

   /**
    * Check if we have attachments enabled.
    * <p/>
    * The check is only done once.
    */
   public static synchronized boolean isAttachmentSupportEnabled()
   {
      return isAttachmentSupportEnabled(null);
   }

   /**
    * Check if we have attachments enabled.
    * <p/>
    * The check is only done once.
    */
   public static synchronized boolean isAttachmentSupportEnabled(MessageContext msgCtx)
   {

      if (checkForAttachmentSupport)
      {
         checkForAttachmentSupport = false;

         try
         {
            if (msgCtx != null)
            {
               AxisEngine engine = msgCtx.getAxisEngine();
               if (engine != null)
               {
                  attachImplClassName = (String)engine.getOption(AxisEngine.PROP_ATTACHMENT_IMPLEMENTATION);
               }
            }

            if (attachImplClassName == null)
            {
               attachImplClassName = AxisEngine.DEFAULT_ATTACHMENT_IMPL;
            }

            /**
             * Attempt to resolve class name, verify that these are present...
             */
            ClassUtils.forName("javax.activation.DataHandler");
            ClassUtils.forName("javax.mail.internet.MimeMultipart");

            attachImplClass = ClassUtils.forName(attachImplClassName);
            attachmentSupportEnabled = true;
         }
         catch (ClassNotFoundException ex)
         {
            // no support for it, leave mAttachments null.
         }
         catch (NoClassDefFoundError ex)
         {
            // no support for it, leave mAttachments null.
         }
         log.debug(Messages.getMessage("attachEnabled") + "  " + attachmentSupportEnabled);
      }

      return attachmentSupportEnabled;
   }

   /**
    * Get the attachment implementation class
    */
   public static Class getImplementationClass()
   {
      return attachImplClass;
   }
}
