/*
 * JBoss, the OpenSource EJB server
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package javax.xml.soap;

import java.io.IOException;
import java.io.InputStream;

/**
 * A factory for creating SOAPMessage objects.
 *
 * A SAAJ client can create a MessageFactory object using the method newInstance, as shown in the following line of code.
 *
 *    MessageFactory mf = MessageFactory.newInstance();
 *
 * A standalone client (a client that is not running in a container) can use the newInstance method to create a MessageFactory object.
 *
 * All MessageFactory objects, regardless of how they are created, will produce SOAPMessage objects that have the following elements by default:
 *
 *    A SOAPPart object
 *    A SOAPEnvelope object
 *    A SOAPBody object
 *    A SOAPHeader object
 *
 * MessageFactory objects can be initialized with a JAXM profile. In such a case it will produce messages that also
 * come prepopulated with additional entries in the SOAPHeader object and the SOAPBody object. The content of a new
 * SOAPMessage object depends on which of the two MessageFactory methods is used to create it.
 *
 *    createMessage() -- message has no content
 *    This is the method clients would normally use to create a request message.
 *
 *    createMessage(MimeHeaders, java.io.InputStream) -- message has content from the InputStream object and headers from the MimeHeaders object
 *    This method can be used internally by a service implementation to create a message that is a response to a request.
 *
 * @author Scott.Stark@jboss.org
 * @version $Revision:1$
 */
public abstract class MessageFactory
{
   private static final String DEFAULT_MSG_FACTORY = "org.jboss.axis.soap.MessageFactoryImpl";

   public MessageFactory()
   {
   }

   /** Creates a new MessageFactory object that is an instance of the default implementation.
    */
   public static MessageFactory newInstance() throws SOAPException
   {
      MessageFactory factory = null;
      String factoryName = null;

      try
      {
         factoryName = System.getProperty("javax.xml.soap.MessageFactory", DEFAULT_MSG_FACTORY);
         ClassLoader loader = Thread.currentThread().getContextClassLoader();
         Class factoryClass = loader.loadClass(factoryName);
         factory = (MessageFactory)factoryClass.newInstance();
      }
      catch (Throwable t)
      {
         throw new SOAPException("Failed to create MessageFactory: " + factoryName, t);
      }
      return factory;
   }

   /**
    * Creates a new SOAPMessage object with the default SOAPPart, SOAPEnvelope, SOAPBody, and SOAPHeader objects.
    * Profile-specific message factories can choose to prepopulate the SOAPMessage object with profile-specific headers.
    *
    * Content can be added to this message's SOAPPart object, and the message can be sent "as is" when a message
    * containing only a SOAP part is sufficient. Otherwise, the SOAPMessage object needs to create one or more
    * AttachmentPart objects and add them to itself. Any content that is not in XML format must be in an AttachmentPart object.
    *
    * @return a new SOAPMessage object
    * @throws SOAPException if a SOAP error occurs
    */
   public abstract SOAPMessage createMessage() throws SOAPException;

   /**
    * Internalizes the contents of the given InputStream object into a new SOAPMessage object and returns the SOAPMessage object.
    *
    * @param headers the transport-specific headers passed to the message in a transport-independent fashion for creation of the message
    * @param in the InputStream object that contains the data for a message
    * @return a new SOAPMessage object containing the data from the given InputStream object
    * @throws IOException if there is a problem in reading data from the input stream
    * @throws SOAPException  if the message is invalid
    */
   public abstract SOAPMessage createMessage(MimeHeaders headers, InputStream in) throws IOException, SOAPException;
}
