/***************************************
 *                                     *
 *  JBoss: The OpenSource J2EE WebOS   *
 *                                     *
 *  Distributable under LGPL license.  *
 *  See terms of license at gnu.org.   *
 *                                     *
 ***************************************/

package org.jboss.util;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.logger.Log4JLogger;
import org.jacorb.config.LoggerFactory;
import java.util.HashMap;
import java.util.Map;

/**
 * JacORB logger factory that creates named Avalon loggers with log4j
 * as the underlying log mechanism.
 * <p>
 * JacORB log priorities for new loggers are set implicitly to either 
 * the value of this factory's <code>defaultPriority</code> field, or 
 * via JacORB configuration properties that have the same name as the 
 * requested logger, plus a suffix of <code>.log.verbosity</code>.
 *
 *  @author <a href="mailto:reverbel@ime.usp.br">Francisco Reverbel</a>
 *  @version $Revision:1$
 */
public class Log4jLoggerFactory
   implements LoggerFactory
{
   /** logging back-end mechanism used by all Log4jLoggerFactory instances */
   private final static String name = "log4j";
   
   /** default priority for loggers created with this factory */
   private int defaultPriority = 0;
   
   /** cache of created loggers */
   private final Map namedLoggers = new HashMap();

   /** object that represents the JacORB configuration properties */
   private Configuration configuration = null;

   // Auxiliary static methods --------------------------------------

   /**
    * Gets a log4j logger by name.
    *
    * @param name the name of the logger
    * @return an <code>org.apache.log4j.Logger</code> instance
    */
   private static org.apache.log4j.Logger getLog4jLogger(String name)
   {
      org.jboss.logging.Logger l = org.jboss.logging.Logger.getLogger(name);
      org.jboss.logging.LoggerPlugin lp = l.getLoggerPlugin();
      if (lp instanceof org.jboss.logging.Log4jLoggerPlugin)
         return ((org.jboss.logging.Log4jLoggerPlugin)lp).getLogger();
      else
         return null;
   }
   
   /**
    * Returns the priority level for a given integer.
    *
    * @param priority an <code>int</code> value
    * @return an <code>org.apache.log4j.Level</code> value
    */
   private static org.apache.log4j.Level intToLevel(int priority)
   {
      switch (priority)
      {
      case 4:
         return org.apache.log4j.Level.DEBUG;
      case 3:
         return org.apache.log4j.Level.INFO;
      case 2:
         return org.apache.log4j.Level.WARN;
      case 1:
         return org.apache.log4j.Level.ERROR;
      case 0:
      default:
         return org.apache.log4j.Level.FATAL;
      }
   }

   // Auxiliary methods ---------------------------------------------

   /**
    * Returns the priority for a named logger. If no priority is
    * specified for the requested name the priority of the first
    * matching parent logger is returned. If there's no parent 
    * logger, this factory's default priority is returned.
    *
    * @param name the name of a logger
    * @return the priority for that logger
    */
   private int getPriorityForNamedLogger(String name)
   {
      String prefix = name;
      
      while (!prefix.equals(""))
      {
         String priorityString = null;
         
         try
         {
            int priorityForLogger = 
               configuration.getAttributeAsInteger( prefix + ".log.verbosity");
            if (priorityForLogger > 4)
               priorityForLogger = 4;
            else if (priorityForLogger < 0)
               priorityForLogger = 0;
            return priorityForLogger;
         }
         catch( ConfigurationException ce )
         {
         }
         
         if (prefix.lastIndexOf(".") >= 0)
         {
            prefix = prefix.substring(0, prefix.lastIndexOf("."));
         }
         else
         {
            prefix = "";
         }
      }
      
      // nothing found, return default
      return defaultPriority;
   }
   
   // Implementation of org.apache.avalon.framework.configuration.Configuration

   public void configure(Configuration configuration)
      throws ConfigurationException
   {
      this.configuration = configuration;
      
      defaultPriority = 
         configuration.getAttributeAsInteger("jacorb.log.default.verbosity", 0);
      switch (defaultPriority)
      {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
         break;
      default:
         throw new ConfigurationException("'" + defaultPriority + "' is an illegal"
                                          + " value for the property jacorb.log.default.verbosity. Valid values are [0-4]");
      }
   }
   
   // Implementation of org.jacorb.util.LoggerFactory ---------------

   /**
    * Gets the name of the logging back-end mechanism.
    *
    * @return the string <code>"log4j"</code> 
    */
   public final String getLoggingBackendName()
   {
      return name;
   }
   
   /**
    * Gets an Avalon logger by name.
    *
    * @param name the name of the logger
    * @return an <code>org.apache.avalon.framework.logger.Logger</code> 
    *         instance
    */
   public Logger getNamedLogger(String name)
   {
      Object o = namedLoggers.get(name);
      
      if (o != null)
         return (Logger)o;
      
      org.apache.log4j.Logger log4jLogger = getLog4jLogger(name);
      int priority = getPriorityForNamedLogger(name);
      log4jLogger.setLevel(intToLevel(priority));
      
      Logger logger = new Log4JLogger(log4jLogger);
      
      namedLoggers.put(name, logger);
      return logger;
   }
   
   /**
    * Gets an Avalon root logger by name.
    *
    * @param name the name of the logger
    * @return an <code>org.apache.avalon.framework.logger.Logger</code> 
    *         instance
    */
   public Logger getNamedRootLogger(String name)
   {
      return getNamedLogger(name);
   }
   
   /**
    * Creates a named Avalon logger with given <code>logFileName</code>
    * and <code>maxLogSize</code> parameters. This is a dummy implementation
    * that always return null.
    *
    * @param name the name of the logger
    * @param logFileName the name of the file to log to
    * @param maxLogSize maximum size of the log file. 
    *
    * @return the new logger (null in this implementation).
    */
   public Logger getNamedLogger(String name,
                                String logFileName, long maxLogSize)
      throws java.io.IOException
   {
      return null;
   }
   
   /**
    * set the file name and max file size for logging to a file
    */ 
   public void setDefaultLogFile(String fileName, long maxLogSize) 
      throws java.io.IOException 
   {
      // not implemented
   }

}
