/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.hibernate.cache;

import org.hibernate.cache.CacheProvider;
import org.hibernate.cache.Cache;
import org.hibernate.cache.CacheException;

import java.util.Properties;

import org.jboss.cache.TreeCache;
import org.jboss.cache.TreeCacheMBean;
import org.jboss.mx.util.MBeanProxy;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.tm.TxManager;
import org.jboss.logging.Logger;

import javax.management.ObjectName;
import javax.management.MBeanServer;

/**
 * A Hibernate CacheProvider implementation which knows how to
 * obtain a deployed JBossCache via its JMX ObjectName.
 *
 * @version <tt>$Revision:1$</tt>
 * @author <a href="mailto:steve@hibernate.org">Steve Ebersole</a>
 */
public class DeployedTreeCacheProvider implements CacheProvider
{
   private static final Logger log = Logger.getLogger( DeployedTreeCacheProvider.class );

   public static final String OBJECT_NAME_PROP = "hibernate.treecache.objectName";
   public static final String DEFAULT_OBJECT_NAME = "jboss.cache:service=HibernateTreeCache";

   private TreeCache deployedTreeCache;

   public void start(Properties properties) throws CacheException
   {
      // Determine the TreeCache MBean ObjectName.
      String configObjectName = properties.getProperty( OBJECT_NAME_PROP, DEFAULT_OBJECT_NAME );
      ObjectName objectName;
      try
      {
         objectName = new ObjectName( configObjectName );
      }
      catch( Throwable t )
      {
         throw new CacheException( "Malformed TreeCache ObjectName");
      }

      TreeCacheMBean mbean;
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         mbean = (TreeCacheMBean) MBeanProxy.get(TreeCacheMBean.class, objectName, server);
      }
      catch( Throwable t )
      {
         log.warn( "Unable to locate TreeCache MBean under object name [" + configObjectName + "]", t );
         throw new CacheException( "Unable to locate TreeCache MBean under object name [" + configObjectName + "]" );
      }

      deployedTreeCache = mbean.getInstance();
   }

   public void stop()
   {
      deployedTreeCache = null;
   }

   public boolean isMinimalPutsEnabledByDefault()
   {
      return true;
   }

   /**
    * Called by Hibernate in order to build the given named cache "region".
    *
    * @param name The cache "region" name.
    * @param properties The configuration properties.
    * @return The constructed Cache wrapper around the jndi-deployed TreeCache.
    * @throws CacheException Generally indicates a problem locating the TreeCache.
    */
   public Cache buildCache(String name, Properties properties) throws CacheException
   {
      return new org.hibernate.cache.TreeCache( deployedTreeCache, name, TxManager.getInstance() );
   }

   public long nextTimestamp()
   {
		return System.currentTimeMillis() / 100;
   }
}
