/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.deployment.spi;

// $Id:LocalhostTarget.java,v 1.0, 2005-06-24 19:10:56Z, Robert Worsnop$

import org.jboss.logging.Logger;

import javax.enterprise.deploy.shared.ModuleType;
import javax.enterprise.deploy.spi.TargetModuleID;
import javax.enterprise.deploy.spi.exceptions.TargetException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * A Target interface represents a single logical core server of one instance of a J2EE platform product.
 * It is a designator for a server and the implied location to copy a configured application for the server to access.
 * 
 * @author thomas.diesler@jboss.org
 * @version $Revision:1$
 */
public class LocalhostTarget implements JBossTarget
{
   // deployment logging
   private static final Logger log = Logger.getLogger(LocalhostTarget.class);

   /**
    * Get the target's description
    * 
    * @return the description
    */
   public String getDescription()
   {
      return "JBoss localhost deployment target";
   }

   /**
    * Get the target's name
    * 
    * @return the name
    */
   public String getName()
   {
      return "localhost";
   }

   /**
    * Get the target's host name
    */
   public String getHostName()
   {
      try
      {
         return InetAddress.getLocalHost().getHostName();
      }
      catch (UnknownHostException e)
      {
         log.error("Cannot obtain localhost", e);
         return null;
      }
   }

   /**
    * Deploy a given module
    */
   public void deploy(TargetModuleID targetModuleID)
           throws Exception
   {
      // get the server deploydir
      String deployDir = getDeploydir();

      FileOutputStream outs = null;
      FileInputStream ins = null;
      try
      {
         File deployableFile = new File(targetModuleID.getModuleID());
         File targetFile = new File(deployDir + "/" + deployableFile.getName());
         log.info("Writing deployableFile: " + deployableFile.getAbsolutePath()
                 + " to: " + targetFile.getAbsolutePath());
         outs = new FileOutputStream(targetFile);
         ins = new FileInputStream(deployableFile);
         JarUtils.copyStream(outs, ins);
         log.info("Waiting 10 seconds for deploy to finish...");
         Thread.sleep(10000);
      }
      finally
      {
         try
         {
            if (outs != null)
            {
               outs.close();
            }
            if (ins != null)
            {
               ins.close();
            }
         }
         catch (IOException e)
         {
            // ignore this one            
         }
      }
   }

   /**
    * Start a given module
    */
   public void start(TargetModuleID targetModuleID) throws Exception
   {
   }

   /**
    * Stop a given module
    */
   public void stop(TargetModuleID targetModuleID) throws Exception
   {
   }

   /**
    * Undeploy a given module
    */
   public void undeploy(TargetModuleID targetModuleID)
           throws Exception
   {
      // get the server deploydir
      String deployDir = getDeploydir();
      File file = new File(deployDir + "/" + targetModuleID.getModuleID());
      file.delete();
   }

   /**
    * Retrieve the list of all J2EE application modules running or not running on the identified targets.
    */
   public TargetModuleID[] getAvailableModules(ModuleType moduleType) throws TargetException
   {
      return null;
   }

   /**
    * Get the server deploydir
    */
   private String getDeploydir()
   {
      //[todo] replace this System property lookup
      String deployDir = System.getProperty("jboss.deploy.dir");
      if (deployDir == null)
      {
         String j2eeHome = System.getProperty("J2EE_HOME");
         if (j2eeHome == null)
            throw new RuntimeException("Cannot obtain system property: jboss.deploy.dir or J2EE_HOME");
         deployDir = j2eeHome + "/server/cts/deploy";
      }
      log.info("Using deploy dir: " + deployDir);
      return deployDir;
   }
}
