/**
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.deployment.spi;

import org.jboss.deployment.spi.status.ProgressObjectImpl;
import org.jboss.logging.Logger;

import javax.enterprise.deploy.shared.CommandType;
import javax.enterprise.deploy.shared.StateType;
import javax.enterprise.deploy.spi.TargetModuleID;
import javax.enterprise.deploy.spi.status.ProgressObject;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

// $Id:DeploymentWorker.java,v 1.0, 2005-06-24 19:01:14Z, Robert Worsnop$

/**
 * A thread that deployes the given deployment on all targets contained in the progress object.
 * It sends events to the progress object.
 * 
 * @author thomas.diesler@jboss.org
 */
public class DeploymentWorker extends Thread
{
   // deployment logging
   private static final Logger log = Logger.getLogger(DeploymentWorker.class);

   private ProgressObjectImpl progress;

   public DeploymentWorker(ProgressObject progress)
   {
      this.progress = (ProgressObjectImpl) progress;
   }

   /**
    * Deploy the module on each given target
    */
   public void run()
   {

      CommandType cmdType = progress.getDeploymentStatus().getCommand();
      TargetModuleID[] modules = progress.getResultTargetModuleIDs();
      for (int i = 0; i < modules.length; i++)
      {
         TargetModuleID moduleid = modules[i];
         JBossTarget target = (JBossTarget) moduleid.getTarget();
         try
         {
            progress.sendProgressEvent(StateType.RUNNING, "Operation " + cmdType + " started", moduleid);
            if (cmdType == CommandType.DISTRIBUTE)
            {
               target.deploy(moduleid);
               deleteDeployment(moduleid);
            }
            else if (cmdType == CommandType.START)
            {
               target.start(moduleid);
            }
            else if (cmdType == CommandType.STOP)
            {
               target.stop(moduleid);
            }
            else if (cmdType == CommandType.UNDEPLOY)
            {
               target.undeploy(moduleid);
               deleteDeployment(moduleid);
            }
            progress.sendProgressEvent(StateType.COMPLETED, "Operation " + cmdType + " completed", moduleid);
         }
         catch (Exception e)
         {
            String message = "Operation " + cmdType + " failed on target " + target;
            progress.sendProgressEvent(StateType.FAILED, message, moduleid);
            log.error(message, e);
         }
      }
   }

   private void deleteDeployment(TargetModuleID moduleid) throws MalformedURLException
   {
      File deployment = new File(new URL(moduleid.getModuleID()).getPath());
      if (deployment.exists())
      {
         if (!deployment.delete())
         {
            log.warn("Cannot delete deployment file " + deployment + ", will be deleted on exit");
            deployment.deleteOnExit();
         }
      }
      else
      {
         log.error("Deployment does not exist: " + deployment);
      }
   }
}
