/*
* JBoss, the OpenSource J2EE webOS
*
* Distributable under LGPL license.
* See terms of license at gnu.org.
*/
package org.jboss.cache.tests;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.TreeCache;
import org.jboss.cache.TreeCacheListener;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.transaction.DummyTransactionManager;
import org.jgroups.View;

import javax.transaction.NotSupportedException;
import javax.transaction.Transaction;

/**
 * Tests whether modifications within callbacks (TreeCacheListener) are handled correctly
 * @author Bela Ban
 * @version $Id:CallbackTest.java,v 1.0, 2005-06-24 18:56:27Z, Robert Worsnop$
 */
public class CallbackTest extends TestCase {
   TreeCache cache=null, cache2;
   Transaction tx=null;
   final Fqn FQN=Fqn.fromString("/myNode");
   final String KEY="key";
   final String VALUE="value";





   Listener exListener=new Listener() {
      public void nodeCreated(Fqn fqn) {
         throw new RuntimeException("this will cause the TX to rollback");
      }
   };

   protected void setUp() throws Exception {
      super.setUp();
   }

   protected void tearDown() throws Exception {
      super.tearDown();
      if(cache != null) {
         cache.stopService();
         cache.destroyService();
         cache=null;
      }
      if(tx != null) {
         tx.commit();
         tx=null;
      }
   }





   public void testLocalCallbackWithoutTransaction() throws Exception, NotSupportedException {
      cache=createCache(TreeCache.LOCAL, IsolationLevel.SERIALIZABLE);
      cache.addTreeCacheListener(new MyListener(cache));

      cache.put("/a", null);
      assertTrue(cache.exists(FQN));
   }


   public void testLocalCallbackWithTransaction() throws Exception, NotSupportedException {
      cache=createCache(TreeCache.LOCAL, IsolationLevel.SERIALIZABLE);
      cache.addTreeCacheListener(new MyListener(cache));
      tx=startTransaction();
      cache.put("/a", null);
      tx.commit();
      assertTrue(cache.exists(FQN));
   }


   public void testLocalCallbackWithException() throws Exception, NotSupportedException {
      cache=createCache(TreeCache.LOCAL, IsolationLevel.SERIALIZABLE);
      cache.addTreeCacheListener(exListener);
      tx=startTransaction();
      try {
         cache.put("/a", null);
         tx.rollback();
      }
      catch(RuntimeException ex) {
        tx.rollback();
      }
      assertFalse(cache.exists(FQN));
      // assertFalse(cache.exists("/a"));  -- fix: reason is that CreateIfNotExists interceptor is *before*
      // LockInterceptor, which registers for TX completion, we don't even get that far
   }

   /*public void testSyncReplicationWith2Caches() throws Exception, NotSupportedException {
      cache=createCache(TreeCache.REPL_SYNC, IsolationLevel.SERIALIZABLE);
      cache2=createCache(TreeCache.REPL_SYNC, IsolationLevel.SERIALIZABLE);
      cache.setSyncCommitPhase(true);
      cache2.setSyncCommitPhase(true);

      // cache.setSyncReplTimeout(60000); // todo: remove again, just for debugging
      // cache2.setSyncReplTimeout(60000); // todo: remove again, just for debugging

      assertEquals(2, cache.getMembers().size());
      System.out.println("view is correct: " + cache.getMembers() + " (2 members)");
      cache2.addTreeCacheListener(new MyListener(cache2));

      tx=startTransaction();
      cache.put("/a", null);
      tx.commit();

      // 1. cache: put("/a")
      // 2. tx.commit() in cache causes 2PC to cache2
      // 3. cache2 is updated with "/a"
      // 4. listener in cache2 is called, creates FQN
      // 5. cache2 replicates FQN over to cache as part of its own 2PC protocol

      // Util.sleep(1000000);

      assertTrue(cache.exists("/a"));  // original modification to cache
      assertTrue(cache.exists(FQN));   // result of replication from cache
      assertTrue(cache2.exists("/a")); // result of replication from cache
      assertTrue(cache2.exists(FQN));  // created by listener, triggered by replication of "/a" from cache
      TransactionTable tx_table1, tx_table2;
      tx_table1=cache.getTransactionTable();
      tx_table2=cache2.getTransactionTable();
      System.out.println("tx_table1=" + tx_table1 + ", tx_table2=" + tx_table2);
      assertEquals(0, tx_table1.getNumLocalTransactions());
      assertEquals(0, tx_table2.getNumLocalTransactions());
      assertEquals(0, tx_table1.getNumGlobalTransactions());
      assertEquals(0, tx_table2.getNumGlobalTransactions());
   }*/



/*   public static void main(String[] args) {
      try {
         new CallbackTest().testSyncReplicationWith2Caches();
      }
      catch(Exception e) {
         e.printStackTrace();
      }
   }*/


   TreeCache createCache(int mode, IsolationLevel level) throws Exception {
      TreeCache c=new TreeCache();
      c.setCacheMode(mode);
      c.setIsolationLevel(level);
      c.setTransactionManagerLookupClass("org.jboss.cache.DummyTransactionManagerLookup");
      c.createService();
      c.startService();
      return c;
   }

   Transaction startTransaction() {
      DummyTransactionManager mgr=DummyTransactionManager.getInstance();
      try {
         mgr.begin();
         return mgr.getTransaction();
      }
      catch(Throwable t) {
         return null;
      }
   }



   class Listener implements TreeCacheListener {

      public void nodeCreated(Fqn fqn) {
      }

      public void nodeRemoved(Fqn fqn) {
      }

      public void nodeLoaded(Fqn fqn) {
      }

      public void nodeEvicted(Fqn fqn) {
      }

      public void nodeModified(Fqn fqn) {
      }

      public void nodeVisited(Fqn fqn) {
      }

      public void cacheStarted(TreeCache cache) {
      }

      public void cacheStopped(TreeCache cache) {
      }

      public void viewChange(View new_view) { // might be MergeView after merging
      }
   }



   class MyListener extends Listener {
      TreeCache c;

      public MyListener(TreeCache c) {
         this.c=c;
      }

      public void nodeCreated(Fqn fqn) {
         try {
            if(!c.exists(FQN)) {
               System.out.println("MyListener: creating node " + FQN);
               c.put(FQN, KEY, VALUE);
               System.out.println("MyListener: created node " + FQN);
            }
         }
         catch(CacheException e) {
            fail("listener was unable to update cache during callback: " + e);
         }
      }

   }

   public static Test suite() {
      return new TestSuite(CallbackTest.class);
   }

   //public static void main(String[] args) {
     // junit.textui.TestRunner.run(suite());
   //}

}
