// <ISS EXTENSION>

#ifdef ISS_EXTENSION_FILTER

namespace ISSExtension
{
//	is extern in XMAIL 1.21	
//	static bool bFilterLogEnabled= true;
	
#define ISS_LOG_MSG(a) { if (bFilterLogEnabled)	::MscFileLog a ; }
	
	static void ISSFilterLogResult(const char* from, DynString & recpt, bool allowed,
		const char* why)
	{
		if (bFilterLogEnabled==false)
			return;
		
		char szTime[256] = "";
		MscGetTimeNbrString(szTime, sizeof(szTime) - 1);
		MscFileLog("issfilter", "\"%s\"\t\"%s\"\t\"%s\"\t\"%s\"\t\"%s\"\n",
			szTime, from, StrDynGet(&recpt), allowed ? "OK" : "REJECTED", why);
	}
	
	class MessageFilter
	{
		static SYS_MUTEX S_mutex;
		// flag if wildcards are supported
		
	public:
		static bool S_WildcardSupport;
		
		static int FilterMessage(SMTPSession & SMTPS, char * & pszError);
		
		static void Log(const char* format, ...);
		
		static void Cleanup();
		
		static void Shutdown();
		
	private:
		static void UpdateQueues(SMTPSession & SMTPS);
		
		static void BuildQueueFromFile(const SMTPSession & SMTPS, const char* path, const char* filename);
		
		static bool FindInQueues(const char* recipient);
		
		struct FilterQueue
		{
			// file the queue was built from
			char _fileName[SYS_MAX_PATH];
			// last modification time of the file at the time of building the queue
			time_t _timeStamp;
			// array of email addresses - sorted ascencing - for quick binary search
			char ** _addresses;
			// number of addressed in _addresses
			size_t _addressCount;
			// flag if found by current scan
			bool _isHit;
			// flag if queue contains wildcarded strings -> performance
			bool _hasWildcards;
			
			FilterQueue()
				: _timeStamp(0)
				, _addresses(0)
				, _addressCount(0)
				, _isHit(false)
				, _hasWildcards(false)
			{
				_fileName[0]= 0;
			}
			
			~FilterQueue()
			{
				Cleanup();
			}
			
			void Cleanup();
			
			void UpdateFromFile(const SMTPSession & SMTPS, const char* path);
			
			static int compare_qsort(const char** left, const char** right)
			{
				return stricmp(*left, *right);
			}
			static int compare_bsearch(const char* left, const char** right)
			{
				return stricmp(left, *right);
			}
			
			bool Find(const char* recipient);
			
		private:
			FilterQueue(const FilterQueue & that);
			const FilterQueue & operator=(const FilterQueue & that);
		};
		
		static FilterQueue ** _queues;
		static size_t _queueCount;
		static size_t _lastUpdate;
	};
	
	SYS_MUTEX MessageFilter::S_mutex= SysCreateMutex();
	bool MessageFilter::S_WildcardSupport= true;
	MessageFilter::FilterQueue ** MessageFilter::_queues= 0;
	size_t MessageFilter::_queueCount= 0;
	size_t MessageFilter::_lastUpdate= 0;
	
	inline 
		static bool MatchString( const char* pattern, const char* inp )
	{
		return 1 == StrIWildMatch(inp, pattern);
	}
	
	inline
		static bool IsWildcardString(const char* s)
	{
		if (s && *s)	// StrIWildMatch supports *[]?
			return strpbrk(s, "*[]?")!=NULL;
		return false;
	}
	
	
	void MessageFilter::Cleanup()
	{
		if (_queues)
		{
			for (size_t i= 0; i<_queueCount; ++i)
				delete _queues[i];
			delete [] _queues;
			_queues= 0;
		}
		_queueCount= 0;
	}
	
	
	
	bool MessageFilter::FilterQueue::Find(const char* recipient)
	{
		if (!_addresses || _addressCount==0 || !recipient || !*recipient)
			return false;
		
		// plain email text search
		if (0 != bsearch(recipient, _addresses, _addressCount, sizeof(_addresses[0]), 
			(int (*)(const void*, const void*))compare_bsearch))
			return true;
		
		if (S_WildcardSupport && _hasWildcards)
		{
			for (size_t i= 0; i<_addressCount; ++i)
			{
				const char* temp= _addresses[i];
				if (!IsWildcardString(temp)) // strpbrk(temp, "*{}[]()+?")==NULL)
					continue;	// no wildcards
				
				if (MatchString(temp, recipient))
					return true;
			}
		}
		return false;
	}
	
	
	void MessageFilter::FilterQueue::UpdateFromFile(const SMTPSession & SMTPS, const char* path)
	{
		SYS_FILE_INFO fi;
		SysGetFileInfo(path, fi);
		if (fi.tMod == _timeStamp)
			return;
		
		Cleanup();
		_timeStamp= fi.tMod;
		
		// read all lines
		FILE* fp= fopen(path, "r");
		if (fp)
		{
			char line[1024]= "";
			const size_t CHUNCK_SIZE= 16;
			size_t alloc_size= 0;
			while (MscGetString(fp, line, sizeof(line)-1))
			{
				StrLTrim(line, " \t<\"");
				StrRTrim(line, " \t>\"");
				
				if (IsEmptyString(line))	// skip empty lines
					continue;
				
				if (IsWildcardString(line))
					_hasWildcards= true;
				
				if (_addressCount >= alloc_size)
				{
					// reallocate and copy
					char ** temp= new char*[alloc_size+= CHUNCK_SIZE];
					memset(temp, 0, alloc_size * sizeof(char*));
					if (_addressCount > 0)
					{
						memcpy(temp, _addresses, _addressCount * sizeof(char*));
						delete [] _addresses;
					}
					_addresses= temp;
				}
				size_t len= strlen(line);
				_addresses[_addressCount]= new char[len+1];
				strcpy(_addresses[_addressCount], line);
				
				++_addressCount;
				*line= 0;
			}
			fclose(fp);
			
			if (_addresses) // sort
			{
				qsort(_addresses, _addressCount, sizeof(char*), 
					(int (*)(const void*, const void*))compare_qsort);
				for (size_t i= 0; i<_addressCount; ++i)
				{
					ISS_LOG_MSG(("issfilter", "\"init added\"\t\"%s\"\n", _addresses[i]))
				}
			}
			ISS_LOG_MSG(("issfilter", "\"init total\"\t\"%d\"\t\"%s\"\n", _addressCount, path));
		}
		else
			Cleanup();
	}
	
	void MessageFilter::Shutdown()
	{
		{
			MutexAutoPtr locker(S_mutex);
			Cleanup();
		}
		SysCloseMutex(S_mutex);
		S_mutex= 0;
		SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter stopped\n");
	}
	
	
	void MessageFilter::FilterQueue::Cleanup()
	{
		if (_addresses)
		{
			for (size_t i= 0; i<_addressCount; ++i)
				delete[] (_addresses[i]);
			delete[] (_addresses);
			_addresses= 0;
		}
		_addressCount= 0;
		_timeStamp= 0;
		_hasWildcards= false;
	}
	
	
	bool MessageFilter::FindInQueues(const char* recipient)
	{
		// if no queues -> do not filter!!!
		if (!_queues)
			return true;
		
		size_t callCount= 0;
		for (size_t i= 0; i<_queueCount; ++i)
		{
			if (_queues[i])
			{
				++callCount;
				if (_queues[i]->Find(recipient))
					return true;
			}
		}
		
		// if no queues -> do not filter
		if (callCount == 0)
		{
			Cleanup();
			return true;
		}
		return false;
	}
	
	void MessageFilter::BuildQueueFromFile(const SMTPSession & SMTPS,
		const char* path, 
		const char* filename)
	{
		char fpath[SYS_MAX_PATH]= "";
		sprintf(fpath, "%s%s%s", path, SYS_SLASH_STR, filename);
		
		// find filename in _queues
		FilterQueue * curr= 0;
		size_t i;
		size_t currPos= 0;
		for (i= 0; i<_queueCount && !curr; ++i)
		{
			if (_queues[i])
			{
				if (stricmp(_queues[i]->_fileName, filename) == 0)
				{
					// found
					curr= _queues[currPos= i];
				}
			}
		}
		bool existed= false;
		if (curr)
		{
			existed= true;
			curr->_isHit= true;
		}
		else
		{
			curr= new FilterQueue;
			curr->_isHit= true;
			strcpy(curr->_fileName, filename);
		}
		curr->UpdateFromFile(SMTPS, fpath);
		
		// empty files are ignored!
		if (curr->_addressCount == 0)
		{
			delete curr;
			if (existed)
				_queues[currPos]= 0;
			
			return;
		}
		if (!existed)
		{
			// insert into _queues
			// search for first empty entry 
			for (i= 0; i<_queueCount; ++i)
			{
				if (_queues[i]==0)
				{
					_queues[i]= curr;
					break;
				}
			}
			// if there was no empty space, append
			if (i == _queueCount)
			{
				// reallocate and copy
				FilterQueue** temp= new FilterQueue*[_queueCount + 1];
				if (_queueCount)
				{
					memcpy(temp, _queues, _queueCount * sizeof(_queues[0]));
					delete[] _queues;
				}
				_queues= temp;
				_queues[_queueCount++]= curr;
			}
		}
	}
	
	void MessageFilter::UpdateQueues(SMTPSession & SMTPS)
	{
		time_t now= time(0);
		if (_lastUpdate != 0 && now - _lastUpdate < 60 * 5)	// parse every five minutes
			return;
		
		_lastUpdate= now;
		
		bool enablemsgfilter= true;
		
/*
		char * enabled= SvrGetConfigVar(SMTPS.hSvrConfig, "EnableSmtpISSFilter");
		if (enabled)
		{
			if (strcmp(enabled, "1")==0 || stricmp(enabled, "true")==0 || stricmp(enabled, "yes")==0)
				enablemsgfilter= true;
			SysFree(enabled);
		}
*/
		if (!enablemsgfilter)
		{
			if (_queues)
			{
				ISS_LOG_MSG(("issfilter", "\"disabled\"\t\"cleaning up\"\n"));
			}
			Cleanup();
			SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter disabled\n");
			return;
		}
		
		if (_queues)
		{
			for (size_t i= 0; i<_queueCount; ++i)
				if (_queues[i])
					_queues[i]->_isHit= false;
		}
		
		// parse directory MAIL_ROOT/issfilter for "*.allowed" files
		char path[SYS_MAX_PATH]= "";
		char filename[SYS_MAX_PATH]= "";
		CfgGetRootPath(path, sizeof(path));
		StrNCat(path, "issfilter", sizeof(path));
		SYS_HANDLE fh= SysFirstFile(path, filename);
		if (fh != SYS_INVALID_HANDLE)
		{
			do
			{
				if (SysIsDirectory(fh) || IsDotFilename(filename))
					continue;
				
				// check for ".allowed" extension
				const char * temp= strstr(filename, ".allowed");
				if (temp)
				{
					size_t len= strlen(filename);
					if (temp == filename + len - 8)
					{
						// found one!
						BuildQueueFromFile(SMTPS, path, filename);
					}
				}
			} while (SysNextFile(fh, filename));
			SysFindClose(fh);
		}
		
		// cleanup all queues that where not found 
		if (_queues)
		{
			size_t deleteCount= 0;
			for (size_t i= 0; i<_queueCount; ++i)
			{
				if (_queues[i])
				{
					if(_queues[i]->_isHit == false)
					{
						delete _queues[i];
						_queues[i]= 0;
						++deleteCount;
					}
				}
				else
					++deleteCount;
			}
			// nothing left, cleanup
			if (_queueCount == deleteCount)
			{
				ISS_LOG_MSG(("issfilter", "\"all queues empty\"\t\"cleaning up\"\n"));
				Cleanup();
			}
		}
/*
		S_WildcardSupport= false;
		char * wildcardstr= SvrGetConfigVar(SMTPS.hSvrConfig, "SmtpISSFilterEnableWildcards");
		if (wildcardstr)
		{
			if (strcmp(wildcardstr, "1")==0 || stricmp(wildcardstr, "true")==0 || stricmp(wildcardstr, "yes")==0)
				S_WildcardSupport= true;
			SysFree(wildcardstr);
		}
		ISS_LOG_MSG(("issfilter", "\"wildcard support\"\t\"%s\"\n", S_WildcardSupport ? "enabled" : "disabled"));
*/
	}
	
	// read the spool file and check for every recipient if it is in the allowed lists
	// -) if at least one recipient is allowed, continue normal processing
	// -) if no recipient is allowed, return error
	int MessageFilter::FilterMessage(SMTPSession & SMTPS, char * & pszError)
	{
		MutexAutoPtr locker(S_mutex);
		
		static int first= true;
		if (first)
		{
			SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter starting\n");
			first= false;
		}
		
		UpdateQueues(SMTPS);
		
		if (!_queues)
			return 0;
		
		ISS_LOG_MSG(("issfilter", "\"%s\"\t\"%s%s\"\t\"checking\"\n", SMTPS.pszFrom, SMTPS.pszRcpt,
			SMTPS.iRcptCount>1 ? "(and others)" : ""));
		
		// iss filtering is skipped if the sending host is a known relay host
		// this is to prevent issfilter to disallow outgoing mails
		bool isAllowedRelay= true;
		bool ok= 0 == USmtpIsAllowedRelay(SMTPS.PeerInfo, SMTPS.hSvrConfig);
		
		DynString recipients;
		StrDynInit(&recipients);
		const char* rec_sep= ";";
		const int rec_sep_len= 1;
		if (ok == false)
		{
			isAllowedRelay= false;
			// special case: only one recipient, fast handling
			if (SMTPS.iRcptCount == 1)
			{
				ok= FindInQueues(SMTPS.pszRcpt);
				StrDynAdd(&recipients, SMTPS.pszRcpt);
			}
			else
			{
				char line[MAX_SPOOL_LINE] = "";
				
				unsigned long pos= ftell(SMTPS.pMsgFile);
				rewind(SMTPS.pMsgFile);
				size_t clen= strlen(RCPT_TO_STR);
				bool foundOne= false;
				while (ok==false && MscGetString(SMTPS.pMsgFile, line, sizeof(line)-1))
				{
					// read the message file
					if (strnicmp(line, RCPT_TO_STR, clen) != 0)
						if (foundOne)
							break;
						else
							continue;
						
						foundOne= true;
						SMTPTrimRcptLine(line);
						char* temp= line+clen;
						temp= StrLTrim(temp, " \t<");
						StrRTrim(temp, ">");
						const char* rectemp= StrDynGet(&recipients);
						if (rectemp && *rectemp)
							StrDynAdd(&recipients, rec_sep, rec_sep_len);
						StrDynAdd(&recipients, temp);
						if (FindInQueues(temp))
							ok= true;
				}
				fseek(SMTPS.pMsgFile, pos, SEEK_SET);
			}
		}
		if (StrDynGet(&recipients)==0)
		{
			StrDynAdd(&recipients, SMTPS.pszRcpt);
		}
		if (ok == false)
		{
			// create custom error message, if empty, XMAIL will return "554 Transaction failed"
			// configure this error message in server.tab
			char* mode= SvrGetConfigVar(SMTPS.hSvrConfig, "ISSPreDataFilterMode", "silent");
			if (mode)
			{
				if (strcmp(mode, "silent")==0)
				{
					SMTPS.pvmailDrop= true;
					SysFree(mode);
					return ERR_FILTERED_MESSAGE;
				}
				SysFree(mode);
			}
			pszError = SvrGetConfigVar(SMTPS.hSvrConfig, "ISSPreDataFilterResponse");
			ISSFilterLogResult(SMTPS.pszFrom, recipients, false, "unknown recipient(s)");
			StrDynFree(&recipients);
			ErrSetErrorCode(ERR_FILTERED_MESSAGE);
			return (ERR_FILTERED_MESSAGE);
		}
		else
		{
			ISSFilterLogResult(SMTPS.pszFrom, recipients, true, 
				isAllowedRelay ? "sender host is allowed relay" : "known recipient(s)");
			StrDynFree(&recipients);
			return 0;
		}
	}
	
#undef ISS_LOG_MSG 
	
}	// namespace ISSExtension
#endif	// ISS_EXTENSION
// </ISS EXTENSION>


