/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */

/*
 * Modified Cobion AG
 * 2002-09-19
 */
 
#include "threadpool.h"
#include "queue.h"
#include <stdio.h>
#include <assert.h>
#ifdef CbWIN32
#include <Windows.h>
#include <errno.h>
#else
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#ifdef CbSOLARIS
#include <errno.h>
#define sem_wait	sema_wait
#define sem_post	sema_post
#define sem_init	sema_init
#else
#include <asm/errno.h>
#endif
#endif
int THREAD_POOL_TERMINATE = 0;
typedef struct pool_thread {
    int         id;
#ifdef CbWIN32
    HANDLE 	    thr;
    HANDLE      mtx;
    HANDLE      cond;
#else
    pthread_t        thr;
    pthread_mutex_t  mtx;
    pthread_cond_t   cond;
#endif 
    void 	    *arg;

    CIRCLEQ_ENTRY(pool_thread)	q_link;
} pool_thread_t;

typedef CIRCLEQ_HEAD(threadpool_s, pool_thread) threadpool_t;

/*
 * PROTOTYPES
 */
static int threadpool_create_threads(void);
static void threadpool_start_thread(void *arg);
static void threadpool_return_thread(pool_thread_t *thr);

/*
 * GLOBALS
 */
static threadpool_t pool;
static int num_used;
static void *(*job_func)(void *);
static int num_thr;
#ifdef CbWIN32
HANDLE available_sem;
static HANDLE tp_mtx;	       /* for coarse-grained locking */
static HANDLE tp_thrsig_mtx;   /* guarantees threads are ready to be used */
#else
sem_t available_sem;
static pthread_mutex_t tp_mtx; /* for coarse-grained locking */
static pthread_mutex_t tp_thrsig_mtx; /* guarantees threads are ready to be used */
#endif

/* Initialize the threads */
int
threadpool_init(int num, void *(*func)(void *))
{
	assert(num >= 0 && func);
	THREAD_POOL_TERMINATE = 0;
	job_func = func;
	num_thr = num;
	num_used = 0;
	/* initially, no threads are available */
#ifdef CbWIN32
	available_sem = CreateSemaphore(NULL, 0, num_thr, NULL);
	tp_mtx = CreateMutex(NULL, FALSE, NULL);
	tp_thrsig_mtx = CreateMutex(NULL, FALSE, NULL);
#else
	sem_init(&available_sem, 0, 0);
	pthread_mutex_init(&tp_mtx, 0);
	pthread_mutex_init(&tp_thrsig_mtx, 0);
#endif
	CIRCLEQ_INIT(&pool);

	if (threadpool_create_threads() < 0) {
		perror("icap threadpool_create_threads");
		return -1;
	}
	return 0;
}

static int
threadpool_create_threads(void)
{
	int i;
	int res  = 0;
	for (i=0; i < num_thr; i++) {
		pool_thread_t *thr;

		thr = (pool_thread_t*)malloc(sizeof(pool_thread_t));
		if (!thr) {
			perror("icap threadpool_create_threads: malloc error");
			return -1;
		}
		memset(thr, '0', sizeof(pool_thread_t));
		
		/* init data fields of thread structure */
		thr->arg = 0;
		thr->id = i+1;
#ifdef CbWIN32
		thr->mtx  = CreateMutex(NULL, FALSE, NULL);
		thr->cond = CreateSemaphore(NULL, 0, 1, NULL);
		
		if ((thr->thr=CreateThread(NULL, 
					  0, 
					  (LPTHREAD_START_ROUTINE)threadpool_start_thread,
					  (void*) thr, 
					  0, 
					  NULL)) == NULL) {
		  perror("icap thread_create error");
		  return -1;
		}
#else
		pthread_mutex_init(&thr->mtx, 0);
		pthread_cond_init(&thr->cond, 0);
		res = pthread_create(&thr->thr,
				   0,
				   (void *(*)(void*))threadpool_start_thread,
				   (void *)thr); 
		if( res != 0 )
		{
			if( res == EAGAIN )
				fprintf(stderr,"icap pthread_create error: not enough system resources to create a process for the new thread. Or more than PTHREAD_THREADS_MAX threads are already active.");
			else
				fprintf(stderr,"icap pthread_create error: %i",res);
			return -1;
		}
#endif
	    PTHREAD_MUTEX_LOCK(tp_mtx);
	    num_used++; /* artificial inflation, but it works */
	    PTHREAD_MUTEX_UNLOCK(tp_mtx);
	}
	return 0;
}

static void
threadpool_start_thread(void *arg)
{
	pool_thread_t *thr;
	
	assert(arg);

	thr = (pool_thread_t*)arg;
	
	for (;;) {
        	
		PTHREAD_MUTEX_LOCK(tp_thrsig_mtx);
		/* put thread in pool */
		threadpool_return_thread(thr);
#ifdef CbWIN32
		PTHREAD_MUTEX_UNLOCK(tp_thrsig_mtx); 
		/* wait to be called into service */
		WaitForSingleObject(thr->cond, INFINITE);
#else
		/* wait to be called into service */
		pthread_cond_wait(&thr->cond, &tp_thrsig_mtx);
		/* MUST BE DONE IMMEDIATELY! */
		PTHREAD_MUTEX_UNLOCK(tp_thrsig_mtx); 
#endif
#ifndef CbWIN32
		if( THREAD_POOL_TERMINATE == 1 ) {
			//sleep(1);
			pthread_exit( NULL);
		}

#endif
		/* do work */
		job_func(thr->arg);
	}
}


void
threadpool_dispatch(void *arg)
{
	pool_thread_t *thr;
	
	assert(arg);

	/* it should be impossible for the list to be empty at this point */
	assert( &pool != (threadpool_t*)(pool.cqh_first));

	PTHREAD_MUTEX_LOCK(tp_thrsig_mtx);
	PTHREAD_MUTEX_LOCK(tp_mtx);
	thr = pool.cqh_first;
	CIRCLEQ_REMOVE(&pool, thr, q_link);
	thr->arg = arg;
	num_used++;
	PTHREAD_MUTEX_UNLOCK(tp_mtx);
	PTHREAD_MUTEX_UNLOCK(tp_thrsig_mtx);

	/* wake the thread up to begin its execution */
#ifdef CbWIN32	
	ReleaseSemaphore(thr->cond, 1, NULL);
#else
	pthread_cond_signal(&thr->cond);
#endif

}

static void
threadpool_return_thread(pool_thread_t *thr)
{
	assert(thr);

	PTHREAD_MUTEX_LOCK(tp_mtx);
	num_used--;
	CIRCLEQ_INSERT_TAIL(&pool, thr, q_link);
#ifdef CbWIN32
	ReleaseSemaphore(available_sem, 1, NULL);
#else
	sem_post(&available_sem);
#endif
	PTHREAD_MUTEX_UNLOCK(tp_mtx);
}


/* caught Control-C, now call cleanup routine 
void
threadpool_signal_handler(int signum)
{
	iserver_logerr("caught control-C");
 	threadpool_shutdown();
}
*/
/* cleanup the threads, synchronization constructs */

void mySleep( unsigned int x )
{
#ifdef _WIN32
	Sleep( x );
#else
	struct timespec sleeptime;
	sleeptime.tv_sec = x / 1000;
	sleeptime.tv_nsec = 1000*1000*(x % 1000);
	nanosleep( &sleeptime, 0 );
#endif
}

void
threadpool_shutdown(void)
{
	pool_thread_t* thr;
	int i = num_thr;
	THREAD_POOL_TERMINATE = 1;
	if (!i)
		return;

	PTHREAD_MUTEX_LOCK(tp_mtx);
	i -= num_used;
	num_thr=0;
	
	for (; i>0; i--) {
		thr = pool.cqh_first;
		CIRCLEQ_REMOVE(&pool, thr, q_link);
#ifdef CbWIN32
		CloseHandle(thr->mtx);
		CloseHandle(thr->cond);		
		TerminateThread(thr->thr, 0);
#else		
		pthread_cond_signal(&thr->cond);
		mySleep( 100 );
		pthread_mutex_destroy(&thr->mtx);
		pthread_cond_destroy(&thr->cond);
#endif
		free(thr);
	}

	PTHREAD_MUTEX_UNLOCK(tp_mtx);
#ifdef CbWIN32
	CloseHandle(tp_mtx);
	CloseHandle(tp_thrsig_mtx);
	CloseHandle(available_sem);
#else
	pthread_cond_signal(&thr->cond);
	pthread_mutex_destroy(&tp_mtx);
	pthread_mutex_destroy(&tp_thrsig_mtx);
	sem_destroy(&available_sem);
#endif
	THREAD_POOL_TERMINATE = 0;
	iserver_loginfo("All owned threads destroyed");
}
