/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */

/*
 * Modified Cobion AG
 * 2002-09-20
 */

#include "stream.h"
#include "iserver.h"
#include "gbuf.h"
#include <assert.h>
#include <stdio.h>

#define STREAM_READ_FAILED -1

int
read_preview_chunked(iserver_conn_t *conn)
{
	int nread, bytes, read_len;
    int retval;
	char *p;
    int buflen = BUFLEN;
    char * buf = (char*)malloc(BUFLEN);
	char *dbuf = 0;

	assert(conn);
	
	nread = 0;
	while (nread <= conn->preview) {
		
		memset(buf, '\0', buflen);
		if ( (bytes = iserver_readline(conn, &buf, &buflen)) <= 0 ) {
			iserver_logmsg(conn,
				       "error reading chunked");
			nread = STREAM_READ_FAILED;
			conn->closed = 1;
			goto done;
		}

		/* chop carriage-return linefeed */
		p = (char *)strstr(buf, CRLF);
		assert( p );
		if (p) {
            if(p==buf) 
                goto done;
			*p = 0;
			*(p+1) = 0;
		}

		/* convert hex chunk size to decimal */
		if (!sscanf(buf, "%x", &conn->chunk_size)) {
			iserver_logmsg(conn, "error converting chunk size");
			nread = STREAM_READ_FAILED;
			goto done;
		}

		if (conn->chunk_size == 0) { 
		        /* last chunk indicator for preview*/
			/* check for "0; ieof\r\n" 
			 * to decide if this is end of the body also
			 */
			if (!strcmp(buf, "0; ieof"))
				conn->short_preview = 1; 
			/* read extra \r\n as per correction to ICAP10 spec */
			retval = iserver_readline(conn, &buf, &buflen);
            if(retval <= 0) {
			    iserver_logerr("read preview chunked CRLF failed!");
			    conn->closed = 1;
			    nread = STREAM_READ_FAILED;
            }
			goto done;
		}

		read_len = (conn->chunk_size > conn->preview) ?
			conn->preview : conn->chunk_size;
	
		dbuf = (char *)malloc(read_len);
		if (!dbuf) {
			iserver_logmsg(conn, "error allocating dbuf");
			goto done;
		}
		memset(dbuf, '\0', read_len);

		while (bytes = READ(conn->skt, dbuf, read_len)) {

            if(bytes <= 0) {
		        iserver_logerr("read preview chunk failed!");
				conn->closed = 1;
				nread = STREAM_READ_FAILED;
				goto done;
            }

            gbuf_concat(conn->content, dbuf, bytes);
			nread += bytes;
			read_len -= bytes;
			
			if (conn->html_content) {
				if (cinfo.opt_verbose)
					iserver_loginfo("CTNT (Size %d): %s", 
					       conn->content->length,
					       conn->content->data);
			}

			if (!read_len) {
				break;
			}

			memset(dbuf, '\0', read_len);
			
			if (bytes == 0) {
				iserver_logmsg(conn, "chunk2:  TCP closed!");
				conn->closed = 1;
				nread = STREAM_READ_FAILED;
				goto done;
			}
		}

		free(dbuf); 
		dbuf = 0;
		free(buf);
		buf = 0;

		if (nread == conn->chunk_size) {
			/* eat trailing \r\n  */
			retval = iserver_readline(conn, &buf, &buflen);
            if(retval <= 0) {
			    iserver_logerr("read body chunk trailer failed!");
			    conn->closed = 1;
			    nread = STREAM_READ_FAILED;
    			goto done;
            }
			break;		
		}
	}

	if (cinfo.opt_verbose)
		iserver_loginfo("Preview size: %d, read: %d", conn->preview, nread);

 done:
	if (buf)
		free(buf);
	if (dbuf)
		free(dbuf);
	return nread;
}

int
read_body_chunked(iserver_conn_t *conn)
{
	int nread = 0;
	int temp = 0;
	int rsize, bytes;
    int retval;
	char *p=0, *dbuf=0;
    int buflen = BUFLEN;
    char * buf = (char*)malloc(BUFLEN);
	
	if (conn->preview > 0) {
	
		assert(conn);
		
		if (conn->content->length < conn->chunk_size ) {
			conn->chunk_size -= conn->content->length;
			goto chunk_partial;
		}
	}
	
	for (;;) {
		/* eat chunk size*/
		retval = iserver_readline(conn, &buf, &buflen);
		if (retval<=0) { 
                        /* TCP closed */
			iserver_logerr("read body chunk size failed!");
			conn->closed = 1;
			nread = STREAM_READ_FAILED;
			goto done;
		}
		if (!buf[0]) { 
                        /* TCP closed */
			iserver_logerr("blank line close readed!");
			conn->closed = 1;
			nread = STREAM_READ_FAILED;
			goto done;
		}

		/* chop carriage-return linefeed */
		p = (char *)strstr(buf, CRLF);
		assert( p );
		if (p) {
            if(p==buf) { /* CRLF  - no chunk size present */
			    iserver_logmsg(conn, "no chunk size present");
			    goto done;
            }
			*p = 0;
			*(p+1) = 0;
		}

		/* convert hex chunk size to decimal */
		if (!sscanf(buf, "%x", &conn->chunk_size)) {
			iserver_logmsg(conn, "error converting chunk size");
			nread = STREAM_READ_FAILED;
			goto done;
		}

		if (conn->chunk_size == 0) { 
		        /* last chunk indicator for msg body */
			/* received "0; ieof\r\n"  or "0\r\n" */

			/* read extra \r\n as per correction to ICAP10 spec */
			retval = iserver_readline(conn, &buf, &buflen);
            if(retval <= 0) {
			    iserver_logerr("read body chunked CRLF failed!");
			    conn->closed = 1;
			    nread = STREAM_READ_FAILED;
            }
			goto done;
		}

		/* eat chunk body */
	chunk_partial:
		rsize = conn->chunk_size;

		dbuf = (char *)malloc(conn->chunk_size);
		if (!dbuf) {
			perror("icap stream malloc error");
			nread = STREAM_READ_FAILED;
			goto done;
		}
		memset(dbuf, '\0', conn->chunk_size);		
		
		while (bytes = READ(conn->skt, dbuf, rsize)) {

            if(bytes <= 0) {
		        iserver_logerr("read body chunk failed!");
				conn->closed = 1;
				nread = STREAM_READ_FAILED;
				goto done;
            }
			nread += bytes;

			if (bytes == 0) { 
                                /* TCP closed */
				if (cinfo.opt_verbose)
					iserver_loginfo("chunk2: TCP closed (partial chunk)!");
				conn->closed = 1;
				nread = STREAM_READ_FAILED;
				goto done;
			}
			
			if(conn->html_content) {
				if (cinfo.opt_verbose)
					iserver_loginfo("CTNT:  %s",dbuf);
			}

			gbuf_concat(conn->content, dbuf, bytes);
			
			rsize-= bytes;
			
			if (!rsize){
				/* eat trailing \n */
				retval = iserver_readline(conn, &buf, &buflen);
                if(retval <= 0) {
			        iserver_logerr("read body chunk trailer failed!");
			        conn->closed = 1;
			        nread = STREAM_READ_FAILED;
    				goto done;
                }
				break;
			}
			
			memset(dbuf, '\0', conn->chunk_size);
		}	
		free(dbuf);
		dbuf=0;
	}
	free(buf);
	buf=0;

 done: 
	if (buf)
		free(buf);
	if (dbuf)
		free(dbuf);
	return nread;
}
