/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */

#include "iserver.h"
#include "threadpool.h"
#include "gbuf.h"
#include "header.h"
#include "stream.h"
#include "api.h"

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <assert.h>
#include <signal.h>

/*
 * Modified Cobion AG
 * 2002-09-21
 */

#ifdef CbWIN32
#include <stddef.h>
#include <Windows.h>
#include <time.h>
#include <errno.h>
#else
#ifdef CbSOLARIS
#include <errno.h>
#else
#include <sys/errno.h>
#endif
#include <netinet/tcp.h> 
#include <sys/time.h>
#include <unistd.h>
#include <semaphore.h>
#endif

/*
 * GLOBALS
 */
#ifdef CbWIN32
extern HANDLE available_sem; 
HANDLE shutdown_sem = 0;
SOCKET srv_socket = 0; 
#else
extern sem_t available_sem;
sem_t shutdown_sem;
pthread_mutex_t time_mtx;
int srv_socket;
#endif


iserver_info_t cinfo;

int G_ICAP_SHUTDOWN = 0;

/* external ICAP message handler */

void IcapLogMsg( const char* msg, int error, int level );
extern G_ICAP_SHUTDOWN;

void observer_map_addconn();
void observer_map_remconn();

/* 
 * PROTOTYPES
 */
static int iserver_parse_body(iserver_conn_t *conn);
static iserver_conn_t *iserver_alloc_conn(void);
static int iserver_shutdown_conn(iserver_conn_t *conn);
static void iserver_free_conn(iserver_conn_t *conn);
static void iserver_reset_conn(iserver_conn_t *conn);

/*
 * FUNCTIONS
 */

int restartNeeded(iserver_info_t _icap_server_settings ) {
   
	if( _icap_server_settings.opt_port != cinfo.opt_port )
		return 1;
	if( _icap_server_settings.opt_num_thr != cinfo.opt_num_thr )
		return 1;
	return 0;
}

int updateNeeded(iserver_info_t _icap_server_settings ) {
        return (
        _icap_server_settings.opt_pers_conn       != cinfo.opt_pers_conn
        || _icap_server_settings.opt_istag_first  != cinfo.opt_istag_first
        || _icap_server_settings.opt_http_header  != cinfo.opt_http_header
        || _icap_server_settings.opt_health_check != cinfo.opt_health_check
        || _icap_server_settings.opt_user_profile != cinfo.opt_user_profile
        || _icap_server_settings.opt_verbose      != cinfo.opt_verbose );
    }

void 
icap_set_options_to_defaults(iserver_info_t* settings) 
{
    if(settings) {
        settings->opt_port                 = 1344;
        settings->opt_num_thr              = 50;
        settings->opt_pers_conn            = 1;
        settings->opt_verbose              = 0;
        settings->opt_health_check         = 0;
        settings->opt_http_header          = ICAP_HTTP_1_0_403_HEADER;
        settings->opt_istag_first          = 1;
        settings->opt_make_path_absolute   = 0;
        settings->opt_reqmod_cache_percent = 0;
        settings->opt_modify_percent       = 100;
        settings->opt_user_profile         = 0;
    }
}
extern char* icapPrintISTag(char* strbuf);
void
iserver_initialize(void)
{
    ///
/*	cinfo.opt_num_thr = 10;
	cinfo.opt_port = 1344;
	
	cinfo.stat_enabled     = 1;
    cinfo.stat_conn_active = 0;
    cinfo.stat_conn_total  = 0;
    cinfo.stat_gen_istag   = 1;
  //  memset(cinfo.stat_istag, 0, sizeof(cinfo.stat_istag));
    icapPrintISTag(cinfo.stat_istag);

    /* get the hostname for via */
	cinfo.stat_hostname = (char*) malloc(2048*sizeof(char));
	memset(cinfo.stat_hostname, '\0', 2048*sizeof(char));
	if( gethostname(cinfo.stat_hostname, 2048) ) {
		iserver_logerr( "ICAP: Could not retrieve hostname for local machine");
	}
}

void SetVerbose( int enabled );

void ApplyConfig( iserver_info_t *cfg )
{
	PTHREAD_MUTEX_LOCK(cinfo.opt_mtx);
	cinfo.opt_health_check = cfg->opt_health_check;
	cinfo.opt_http_header = cfg->opt_http_header;
	cinfo.opt_istag_first = cfg->opt_istag_first;
	cinfo.opt_make_path_absolute = cfg->opt_make_path_absolute;
	cinfo.opt_num_thr = cfg->opt_num_thr;
	cinfo.opt_pers_conn = cfg->opt_pers_conn;
	cinfo.opt_port = cfg->opt_port;
	cinfo.opt_user_profile = cfg->opt_user_profile;
	cinfo.opt_verbose = cfg->opt_verbose;	
	cinfo.stat_gen_istag = cfg->stat_gen_istag;
	cinfo.stat_enabled = cfg->stat_enabled;
	strcpy( cinfo.stat_istag, cfg->stat_istag );
	strcpy(cinfo.logpath, cfg->logpath );

	SetVerbose( cinfo.opt_verbose );

	PTHREAD_MUTEX_UNLOCK(cinfo.opt_mtx);	
}


int
iserver_await_connections(void)
{
	int result = 0;
	iserver_conn_t *lconn;

	int allow_addr_reuse=1, temp=0, no_delay=1;
#ifdef CbWIN32
    struct linger ling_opt = { 1, 1 };
	WSADATA wsaData;
	shutdown_sem = CreateSemaphore(NULL, 0, 1, NULL);
	if ((WSAStartup(MAKEWORD(2,2), &wsaData)) != 0)
		iserver_logerr("Winsock init error\n");
	cinfo.stat_mtx = CreateMutex(NULL, FALSE, NULL);
	cinfo.opt_mtx = CreateMutex(NULL, FALSE, NULL);
	cinfo.stat_istag_mtx = CreateMutex(NULL, FALSE, NULL);
#else
	
	sem_init(&shutdown_sem, 0, 0);
	pthread_mutex_init(&time_mtx, 0);
	pthread_mutex_init(&cinfo.stat_mtx, 0);
	pthread_mutex_init(&cinfo.opt_mtx, 0);
	pthread_mutex_init(&cinfo.stat_istag_mtx, 0);
#endif

	iserver_initialize();
	iserver_loginfo( "Starting Proventia Web Filter ICAP Server...");

	/* initialize threadpool */
	if (threadpool_init(cinfo.opt_num_thr, 
	    (void *(*)(void *))iserver_connected) < 0) {
		iserver_logerr("Could not allocate threadpool");
		goto error;
	}

	iserver_loginfo( "Pre-created %d threads", cinfo.opt_num_thr);
	
	/* NOTE: SIGINT does not work for windows */
	/* register the signal handler
	if ((*signal)(SIGINT, (void (*)(int))threadpool_signal_handler) == SIG_ERR) { 
		iserver_loginfo("registering ctrl-C sig handler"); 
		return -1; 
	} */
	
	/* ignore SIGPIPE to prevent iserver aborts in linux */
#ifndef CbWIN32
	if (sigignore(SIGPIPE) == -1) {
		iserver_logerr("registering SIGPIPE sig handler");
		return -1;
	}
#endif
	/* setup listen socket */
	lconn = iserver_alloc_conn();
	if (!lconn) {
		iserver_logerr("Could setup listening socket");
		goto error;
	}
	assert(lconn->content->blk_size);
	
	/* get socket */
	srv_socket = socket(AF_INET, SOCK_STREAM, 0);
#ifdef CbWIN32
	if (srv_socket == INVALID_SOCKET) {
		iserver_logerr("socket() failed with error %d",WSAGetLastError());
		goto error;
	}
#else
	if (srv_socket < 0) {
		iserver_logerr("Socket Error");
		goto error;
	}
	fcntl(srv_socket, F_SETFD, FD_CLOEXEC ); 
#endif
	/* 
	 * allow server-side addresses to be reused (don't have
	 * to wait for that annoying timeout)
	 */
	temp = setsockopt(srv_socket,
			  SOL_SOCKET,
			  SO_REUSEADDR,
			  (void *)&allow_addr_reuse,
			  sizeof(allow_addr_reuse));
#ifdef CbWIN32
	if (temp != 0)
#else
    if (temp < 0)
#endif
    {
        iserver_logerr("setsockopt error(1)");
		goto error;
	}
	/* 
	 * turn off the Nagle's algorithm
	 */
	temp = setsockopt(srv_socket,
			  IPPROTO_TCP,
			  TCP_NODELAY,
			  (void *)&no_delay,
			  sizeof(no_delay));
#ifdef CbWIN32
	if (temp != 0) 
#else
    if (temp < 0)  
#endif
    {
		iserver_logerr("setsockopt error(2)");
		goto error;
	}
	/* bind unnamed socket */
	lconn->addr.sin_family      = AF_INET;
	lconn->addr.sin_addr.s_addr = htonl(INADDR_ANY);
	lconn->addr.sin_port        = htons(cinfo.opt_port);
	
	temp = bind(srv_socket, (struct sockaddr*)&lconn->addr, 
		    sizeof(lconn->addr)); 
#ifdef CbWIN32
	if (temp != 0)
#else
	if (temp < 0) 
#endif 
    {
		iserver_logerr("bind error");
		goto error;
	}

	/* accept connections */	
	iserver_loginfo( "Proventia Web Filter ICAP Server started!");
	for (;;) {
		iserver_conn_t *cconn = iserver_alloc_conn();
		cconn->addr_len = sizeof(cconn->addr);
		
		/* socket needs to listen for connections*/
		temp = listen(srv_socket, SOMAXCONN);

	#ifdef CbWIN32
		if (temp != 0)
	#else
		if (temp < 0) 
	#endif
		{
			iserver_logerr("listen error");
			goto error;
		}        
		
		/* wait for available thread */
#ifdef CbWIN32
		WaitForSingleObject(available_sem, INFINITE);
#else
		sem_wait(&available_sem);
#endif
		PTHREAD_MUTEX_LOCK(cinfo.stat_mtx);
		cinfo.stat_conn_total++;
        if(cinfo.stat_conn_total > 100) {
            cinfo.stat_gen_istag++;
            cinfo.stat_conn_total = 0;
        }
        if(cinfo.stat_enabled==0) {
    	    PTHREAD_MUTEX_UNLOCK(cinfo.stat_mtx);
            goto error;
        }
	    PTHREAD_MUTEX_UNLOCK(cinfo.stat_mtx);

		cconn->skt = accept(srv_socket,
				    (struct sockaddr*)&cconn->addr,
				    &cconn->addr_len);
#ifdef CbWIN32
		if (cconn->skt == INVALID_SOCKET)
#else
		if (cconn->skt < 0) 
#endif
        { 
			if( G_ICAP_SHUTDOWN == 0 )
				iserver_logerr("accept error");
			iserver_free_conn(cconn);
			goto error;
		}
		if( G_ICAP_SHUTDOWN == 1 )
		{
			goto error;
		}
#ifdef CbUNIX
fcntl(cconn->skt, F_SETFD, FD_CLOEXEC );
#endif        
		threadpool_dispatch( (void *)cconn );

	}
	
  error:
	iserver_loginfo("Shutdown Proventia Web Filter ICAP Server...");
	threadpool_shutdown();
	iserver_loginfo("Closing Listen port");
	if (lconn)
		iserver_free_conn(lconn);
#ifdef CbWIN32
	CloseHandle(cinfo.stat_mtx);	
	CloseHandle(cinfo.opt_mtx);	
	CloseHandle(cinfo.stat_istag_mtx);	
	if (srv_socket != INVALID_SOCKET)
		closesocket(srv_socket);

    ReleaseSemaphore(shutdown_sem, 1, NULL);
	CloseHandle(shutdown_sem);
	WSACleanup();
#else
	pthread_mutex_destroy(&cinfo.stat_mtx);
	pthread_mutex_destroy(&cinfo.opt_mtx);
	pthread_mutex_destroy(&cinfo.stat_istag_mtx);
	if (srv_socket != -1)
		close(srv_socket);

    sem_post(&shutdown_sem);
	sem_destroy(&shutdown_sem);
#endif
	iserver_loginfo("Proventia Web Filter ICAP Server inactive!");
	return 0;
}

static void
iserver_send_modified_response(iserver_conn_t* conn)
{
	ca_t* response= 0;

	if (conn->icap_req_type == ICAP_REQMOD) {
		/* call reqmod api* with 
		   1) icap hdr + client hdr + entire post msg body)
		*/ 
		if (!conn->icap_reqmod_post)
			assert(conn->content->length == 0);
		if (conn->icap_reqmod_post && !conn->content->length)
			conn->content->length = -1;

		response = api_modify_reqmod( 
              conn->req_hdr->data
            , conn->client_hdr->data
            , conn->content->data
            , conn->content->length
            , conn->x_auth_user
            , conn->x_client_ip );
	} 
	
	assert(response);
	assert(response->beg);
	iserver_writereply(conn, response);
	free(response->beg);
	free(response);
} 	

/* entry point for all client-handling threads
 */
void
iserver_connected(iserver_conn_t *conn)
{
	unsigned int temp=0;

#ifdef CbWIN32
	__try {
#endif
/* TODO: HPG: add thread to observer map */
	// observer_map_addconn();

	/* don't need to lock, as add/sub is idempotent */
	cinfo.stat_conn_active++	;

    if (cinfo.opt_verbose)
		iserver_loginfo("new connection, current active: %d"
        , cinfo.stat_conn_active);

	while (1) {

		/* parse ICAP request header */
		if (header_read_icap(conn) < 0) {
			goto done;
		}

		if (conn->icap_req_type != ICAP_OPTIONS) {
			
			/* parse client request header */
			if (header_read_client(conn) < 0) {
				goto done;
			}
			
			if ( conn->icap_reqmod_post ) {
			
				/* parse message body for preview (if any) */
				if (iserver_parse_body(conn) < 0) {
					goto done;
				}
			}
		}		
		else {
			/* send an options response */
		
			char* options_reply;
		
			/* FUNCPTR*/
			options_reply = api_options_response(conn->icap_uri);
			assert(options_reply);
	        iserver_writeline(conn, options_reply);
			free(options_reply);
            conn->closed = 1; /* close connection after option reply */
			goto done;
		}
		
		/* call appropriate API function to decide the 
		   preview response (if any) */	
		if (conn->preview >= 0) {
			char* response;
			
			if (conn->icap_req_type == ICAP_REQMOD) {
				/* call reqmod api with 
				   1) icap hdr + client hdr + post msg body
				   (max size = preview)
				   2) preview length
				*/
				if (!conn->icap_reqmod_post)
					assert( conn->content == NULL);
			
				response = api_preview_reqmod(conn->req_hdr->data,
								     conn->client_hdr->data,
								     conn->content->data,
								     conn->content->length,
								     conn->preview);
			} 
		
			/* 
			 *  1) send 100 Continue or 204 depending on
			 *     what api returned ( NULL - 100, 
			 *      non-NULL - write out data recvd, free mem and
			 *                 close connection
			 *  2) if (total msg body < preview) && (100 continue was sent)
			 *     send modified hdrs+content (from char* returned) by 
			 *     calling same api as the one used below
			 */
		
			if (response) {
				temp = iserver_writeline(conn, response);
				free(response);
				if (temp<0)
					goto done;
			}
			else {
				if (!conn->short_preview) {
					temp = iserver_writeline(conn, 
					      "ICAP/1.0 100 Continue\r\n\r\n");
					if (temp<0)
						goto done;
				}

				if (conn->icap_reqmod_post && conn->short_preview) {
					iserver_send_modified_response(conn);
				}
			}

			/* total msg body < preview or 204 sent */ 
			if (response || conn->short_preview) 
				goto done;

		}
		
		/* get remaining msg body */
		if ( conn->icap_reqmod_post ) {
			
	/*		if (read_body_chunked(conn) < 0)*/
			iserver_send_modified_response(conn);
			goto done;
		}

		/* send to api: initial+remaining msg body (if preview >= 0) 
		 * or total body (if preview <0)
		 */
		iserver_send_modified_response(conn);
		
	  done:
        if (cinfo.opt_pers_conn && !conn->closed) {
			/* keep connection open for next request */
			iserver_reset_conn(conn);
            conn->request_count++;
			if (cinfo.opt_verbose)
				iserver_loginfo("recycling current connection (requests accepted: %d)"
                , conn->request_count);
		}
		else {
			/* close connection */
			break;
		}
    }
    iserver_shutdown_conn(conn);
	cinfo.stat_conn_active--;
//    cinfo.stat_request_count += conn->request_count;
    if (cinfo.opt_verbose)
	    iserver_loginfo("shutdown connection after %d requests, current active: %d"
            , conn->request_count, cinfo.stat_conn_active);

	iserver_free_conn(conn); 

#ifdef CbWIN32
	}
	__except( EXCEPTION_EXECUTE_HANDLER ) {
		iserver_logerr( "Exception occured in iserver_connected!" );
	}
#endif
// TODO: hpg: remove thread from observer map	
// observer_map_remconn();	

}

/* parse the body of the transmission
 */
static int
iserver_parse_body(iserver_conn_t *conn)
{
	assert(conn);

	return read_body_chunked(conn);
}

/* write (null-terminated) string str to the connection represented by conn */
int
iserver_writeline(iserver_conn_t *conn, const char *str)
{
	unsigned int bytes = 0;

	assert(conn && str);
	
	bytes = WRITE(conn->skt, str, strlen(str));

#ifndef CbWIN32	
	if (errno == SIGPIPE) {
		iserver_logmsg(conn, "TCP connection closed");
		conn->closed=1;
		return -1;
	}
#endif

	if (bytes < 0) {
		iserver_logerr("socket write error");
		return bytes;
	}

	assert(bytes == strlen(str));
	return bytes;
}

/* write ca_t to the connection represented by conn */
int
iserver_writereply(iserver_conn_t *conn, const ca_t *str)
{
	int written = 0;
	
	assert(conn && str);
	
	/* Sanity checks */
	assert(strstr(str->beg, "ICAP/1.0") || 
	       strstr(str->beg, "HTTP"));
	assert(strstr(str->beg, "ISTAG"));

	while (written < (str->len-1)) {
		written += WRITE(conn->skt, str->beg, (str->len-1));
		if (errno == EPIPE) {
			iserver_logmsg(conn, "TCP connection closed");
			conn->closed=1;
			return -1;
		}
		if (written < 0) {
			iserver_logerr("socket write error");
			return written;
		}
	}

    if (cinfo.opt_verbose) {
		iserver_loginfo("bytes written at iserver_writereply: %d", written);
		iserver_loginfo("Reply written:\n%s", str->beg);
	}
	return written;
}

/* 
 * read a line from the connection represented by conn into buf, 
 * return num bytes read
 */
int
iserver_readline(iserver_conn_t *conn, char **buf, int* buflen)
{
	char c;
	int i, retval;
	
	assert(conn && buf && *buf);

	i = 0;
	for (;;) {

        if ( (retval = READ(conn->skt, &c, 1)) <= 0)
            break;
		
		(*buf)[i++] = c;
		if (c == '\n')
			break;

        if (i >= *buflen) {
            char * tmp = (char*)malloc((*buflen)+BUFLEN);
            if(tmp==0) {
            	buf[0] = 0;
        		iserver_logerr("allocate readline buffer failed");
        		return -1;
            }
            memcpy(tmp, *buf, *buflen);
            free(*buf);
            (*buflen) += BUFLEN;
            *buf = tmp;
        }
	}
	if (retval < 0) {
    	buf[0] = 0;
		iserver_logerr("socket read error");
		return retval;
	}
	(*buf)[i] = 0;
	return i;
}

/* allocate an iserver_conn_t, set defaults, and return it */
static iserver_conn_t *
iserver_alloc_conn(void)
{
	iserver_conn_t *c= 0;
	c = (iserver_conn_t *)malloc(sizeof(iserver_conn_t));
	if (!c) {
		iserver_logerr("icap connect malloc error");
		return 0;
	}
	memset(c, '\0', sizeof(iserver_conn_t));
	
	c->chunked = 0;
	c->chunk_size = 0;
	c->content_len = -1;
	c->resp_code = 0;
	c->html_content = 0;
	c->closed = 0;
	c->preview = -1;
	c->short_preview = 0;
	c->icap_req_type = -1;
	c->icap_uri = 0;
	c->icap_reqmod_post = 0;
	c->origin_res_hdr_exist = 0;
	c->origin_res_body_exist = 0;
  	c->request_count = 0;
	c->x_auth_user = 0;
	c->x_client_ip = 0;
    
	if (! (c->content = gbuf_alloc(CONTENT_BLK_SIZE)) ) {
		iserver_logerr("gbuf allocation");
		free(c);
		return 0;
	}
	if (! (c->req_hdr = gbuf_alloc(CONTENT_BLK_SIZE)) ) {
		iserver_logerr("gbuf allocation");
		gbuf_free(c->content);
		free(c);
		return 0;
	}
	if (! (c->client_hdr = gbuf_alloc(CONTENT_BLK_SIZE)) ) {
		iserver_logerr("gbuf allocation");
		gbuf_free(c->content);
		gbuf_free(c->req_hdr);
		free(c);
		return 0;
	}
	if (! (c->origin_hdr = gbuf_alloc(CONTENT_BLK_SIZE)) ) {
		iserver_logerr("gbuf allocation");
		gbuf_free(c->content);
		gbuf_free(c->req_hdr);
		gbuf_free(c->client_hdr);
		free(c);
		return 0;
	}

	return c;
}

/* free an iserver_conn_t */
static int
iserver_shutdown_conn(iserver_conn_t *conn)
{
    // char readBuffer[1025];

#ifdef CbWIN32   
    if(shutdown(conn->skt, SD_SEND) == SOCKET_ERROR)
#else
    if(shutdown(conn->skt, 1) < 0)
#endif     
    {
   		iserver_logerr("shutdown connection failed");
        return -1;
    }

	/*
	//Do this to be more robust against erroneous clients
    while (1)
	{
		//Non-blocking select
		struct timeval t={0,0};
		struct fd_set set;
		int cnt, newBytes;
		FD_ZERO(&set);
		FD_SET(conn->skt,&set);
		cnt=select(1,&set,NULL,NULL,&t);
		if(!cnt) break;

        newBytes = READ(conn->skt, readBuffer, 1024);
#ifdef CbWIN32   
        if(newBytes == SOCKET_ERROR)
#else
        if(newBytes <0) 
#endif
        {
             return -2;
        } else if(newBytes !=0) {
            if(cinfo.opt_verbose) {
                readBuffer[newBytes]=0;
                iserver_loginfo("unexpected bytes: %d\n%s\n", newBytes, readBuffer);
            }
        } else {
            break;
        }
    }
	*/

#ifdef CbWIN32   
    shutdown( conn->skt, SD_BOTH );    
	if(closesocket(conn->skt) == SOCKET_ERROR) {
   		iserver_logerr("close connection failed");
        return -3;
    }
#else
/*	fflush(NULL); */
	shutdown( conn->skt, SHUT_RDWR );
	close(conn->skt);
#endif
//	iserver_free_conn(conn);
    return 1;
}

/* free an iserver_conn_t */
static void
iserver_free_conn(iserver_conn_t *conn)
{
	assert (conn);
	
	if (conn->icap_uri)
        free(conn->icap_uri);
	conn->icap_uri = 0;
	
	if (conn->req_hdr)
		gbuf_free(conn->req_hdr);

	if (conn->client_hdr)
		gbuf_free(conn->client_hdr);

	if (conn->origin_hdr)
		gbuf_free(conn->origin_hdr);

	if (conn->content)
		gbuf_free(conn->content);

    if (conn->x_auth_user)
	{
        free(conn->x_auth_user);
		conn->x_auth_user= 0;
	}

    if (conn->x_client_ip)
    {
		free(conn->x_client_ip);
		conn->x_client_ip= 0;
	}

	free(conn);
}

/* reset an iserver_conn_t
 */
static void
iserver_reset_conn(iserver_conn_t *conn)
{
	assert(conn);

	conn->chunked = 0;
	conn->chunk_size = 0;
	conn->content_len = -1;
	conn->resp_code = 0;
	conn->html_content = 0;
	conn->closed = 0;
	conn->preview = -1;
	conn->short_preview = 0;
	conn->icap_req_type = -1;
	conn->icap_uri = 0;
	conn->icap_reqmod_post = 0;
	
    if (conn->icap_uri)
        free(conn->icap_uri);
	conn->icap_uri = 0;

    if (conn->x_auth_user)
        free(conn->x_auth_user);
	conn->x_auth_user = 0;

    if (conn->x_client_ip)
        free(conn->x_client_ip);
	conn->x_client_ip = 0;
	
	gbuf_zero(conn->req_hdr);
	gbuf_zero(conn->client_hdr);
	gbuf_zero(conn->origin_hdr);
	gbuf_zero(conn->content);

	return;
}

void
iserver_loginfo( const char *fmt,...)
{
	char buf[32768];
	va_list ap;
	assert(fmt);
	va_start(ap,fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	IcapLogMsg( buf, 0, 4 );
}

void
iserver_logerr( const char *fmt,...)
{
	char buf[32768];
	va_list ap;
	assert(fmt);
	va_start(ap,fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	IcapLogMsg( buf, 1, 0 );
}

void
iserver_logmsg(iserver_conn_t *conn, const char *fmt,...)
{
	char buf[32768];
	va_list ap;
	assert(fmt);
	va_start(ap,fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	IcapLogMsg( buf, 2, 2 );
}