/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */

 /*
 * Modified Cobion AG
 * 2003-03-07
 */

#include "header.h"
#include "iserver.h"
#include "gbuf.h"
#include "auth.h"
#include <stdio.h>
#include <assert.h>

#ifdef CbWIN32
#include <errno.h>
#else
#ifdef CbSOLARIS
#include <errno.h>
#else
#include <sys/errno.h>
#endif
#endif /* Unix/Windows */

/* Parse the ICAP header */
int
header_read_icap(iserver_conn_t *conn)
{
	int retval, icap_req_recognized=0;
    char *line;
    int buflen = BUFLEN;
    char * buf = (char*)malloc(BUFLEN);

	assert(conn);

	for (;;) {
		memset(buf, '\0', buflen);

		retval = iserver_readline(conn, &buf ,&buflen);
		if (retval < 0) {
			if (errno == EPIPE) { /* 32 */ 
				iserver_logmsg(conn, "TCP connection closed");
			}
			conn->closed = 1;
			goto header_error;

		} else if (retval == 0)
			break;

		line = buf;

		if (line) {

			if (strncmp(line, CRLF, 2)) {
				
				if (cinfo.opt_verbose)
					iserver_loginfo("Server-rhdr : %s", line);
				
				if (strstr(line, "res-hdr")) {
					conn->origin_res_hdr_exist = 1;
				} 
				if (strstr(line, "res-body")) {
					conn->origin_res_body_exist = 1;
				}
				if (!strncmp(line, "OPTIONS", 7)) {
					
					conn->icap_req_type = ICAP_OPTIONS;
					icap_req_recognized=1;

				} else if (!strncmp(line, "REQMOD", 6)) {
				
					conn->icap_req_type = ICAP_REQMOD;
					icap_req_recognized=1;

				} else if (!strncmp(line, "Connection: close", 17)) {
            		conn->closed = 1;
					line = 0;
				} else if (!strncmp(line, "X-Client-IP: ", 13)) {

                    if(conn->x_client_ip) 
                        free(conn->x_client_ip);
                    conn->x_client_ip = (char*)malloc(strlen(line+13)+1);
                    strcpy(conn->x_client_ip, line+13);

                } else if (!strncmp(line, "X-Authenticated-User: ", 22)) {

                    if(conn->x_auth_user) 
                        free(conn->x_auth_user);
                    conn->x_auth_user = icap_get_user_uuencoded(line+22);

				} else if (!strncmp(line, "Transfer-Encoding: chunked", 26)) {
					iserver_logmsg(conn, "Transfer-Encoding field cannot exist for ICAP 1.0");
					goto header_error;
				} else if (!strncmp(line, "Preview:", 8)) {
					if (!sscanf(line, "Preview:%d", &conn->preview))
						conn->preview = 0;
					line = 0;
				} else if (!strncmp(line, "TE:", 3)) {
					/* NO-OP */
				} else {
					/* Dont want no headers besides 
					   one's we recognize */
					line = 0;
				}
			}
			
			/* add to request header */
			gbuf_concat(conn->req_hdr, line, retval);
			
			if ( line && !strncmp(line, CRLF, 2)) {
				if (cinfo.opt_verbose)
					iserver_loginfo("Server-RXXX : %s", line);
				break;
			}
		}
	}

    free(buf);
    buf=0;

    if (icap_req_recognized) {
		return 1;
    } else {
        if (!retval) {
            if(!cinfo.opt_pers_conn) /* persistent connection will be closed by client normally */
			    iserver_logmsg(conn, "conn closed by client");
        } else
			iserver_logmsg(conn, "unrecognized ICAP request");
		
		conn->closed=1;
		return -1;
	}
	
  header_error:
    if(buf)
        free(buf);
	iserver_logmsg(conn, "error while parsing ICAP header");
	return -1;
}

/* parse HTTP client header */
int
header_read_client(iserver_conn_t *conn)
{
	int retval;
    int buflen = BUFLEN;
	char *user_tmp;
    char * buf = (char*)malloc(BUFLEN);

	for (;;) {
		char *line;
		
		memset(buf, '\0', buflen);
		retval = iserver_readline(conn, &buf, &buflen);

		if (retval < 0) {
			if (errno == EPIPE) { /* 32 */ 
				iserver_logmsg(conn, "TCP connection closed");
			}
			conn->closed = 1;
			goto header_error;

		} else if (retval == 0)
			break;

		line = buf;

		if (line) {
			if (!strncmp(line, "POST", 4) && 
			    (conn->icap_req_type == ICAP_REQMOD)) {
				conn->icap_reqmod_post = 1;
    		} 
			else if (conn->x_client_ip==0 && !strncmp(line, "X-Forwarded-For: ", 17)) {
				conn->x_client_ip = (char*)malloc(strlen(line+17)+1);
                strcpy(conn->x_client_ip, line+17);
            }
			else if ( conn->x_auth_user==0 && !strncmp(line, "Proxy-Authorization: ", 21)) {
                char* auth = 0;
				if( auth = strstr(line, "Basic") )
                    conn->x_auth_user = icap_get_user_basic(auth+5);
                else if( auth = strstr(line, "NTLM")) 
                    if( conn->x_client_ip ) {
						user_tmp = icap_get_user_ntlm(auth+4);
						icap_add_usercache_entry( user_tmp , conn->x_client_ip );
						free(user_tmp);
					}
					else
						conn->x_auth_user = icap_get_user_ntlm(auth+4);
    		}
			else if( conn->x_auth_user == 0 && !strncmp( line, "X-ISAAuth-User: ", 16 )) {				
				conn->x_auth_user = icap_get_isa_user( line + 16 );
			}

			gbuf_concat(conn->client_hdr, line, retval);
			
			if (cinfo.opt_verbose)
				iserver_loginfo("Server-chdr : %s", line);
			
			if (!strncmp(line, CRLF, 2))
				break;
		}
	}
    free(buf);
	
	if( conn->x_auth_user == NULL )
		conn->x_auth_user = icap_get_usercache_entry( conn->x_client_ip );
	return 1;

  header_error:
    free(buf);
	iserver_logmsg(conn,"error while parsing HTTP client request hdr");
	return -1;
}

