/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */

#include "gbuf.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>

/* allocate and return a gbuf with block size blk_size */
gbuf_t *
gbuf_alloc(int blk_size)
{
	gbuf_t *buf;

	assert(blk_size > 0);

	buf = (gbuf_t *)malloc(sizeof(gbuf_t));
	if (!buf)
		return 0;
	buf->blk_size = blk_size;
	buf->length = 0;

	buf->data = (char *)malloc(buf->blk_size);
	memset(buf->data, '\0', buf->blk_size);

	return buf;
}

/* zero out a gbuf for reuse */
void
gbuf_zero(gbuf_t *buf)
{
	assert(buf);

	if (buf->data)
		free(buf->data);

	buf->length = 0;
	buf->data = (char *)malloc(buf->blk_size);
	memset(buf->data, '\0', buf->blk_size);
	
	return;
}

/* free a gbuf */
void
gbuf_free(gbuf_t *buf)
{
	assert(buf);

	if (buf->data)
		free(buf->data);

	free(buf);
	return;
}

/* concatenate string str onto buf, allocating more space
 * if necessary */
int
gbuf_concat(gbuf_t *buf, const char *str, int len)
{
	int bs = buf->blk_size;
	int space;
	
	assert(buf);

	if (!str)
		return 1;

	space = bs - (buf->length % bs);

	/* simple case, less than block size; just copy and return success */
	if (space > len) {

		memcpy(buf->data+buf->length, str, len);
		buf->length += len;
		buf->data[buf->length] = '\0';
		return 1;

	} else {
		char *p;
		int pages_req, bytes_req;
		int total_bytes = len + buf->length;
		
		buf->blk_size *= 2;
		pages_req = (total_bytes/buf->blk_size) + 1;
		bytes_req = pages_req * buf->blk_size;
		
		p = (char *)malloc(bytes_req);
		if (!p)
			return 0;
		memset(p, '\0', bytes_req);
		/* copy orig. bytes */
		memcpy(p, buf->data, buf->length);
		/* copy new bytes */
		memcpy(p+buf->length, str, len);
		free(buf->data);
		buf->length += len;
		buf->data = p;
		return 1;
	}
	return 0;
}
