/*
 * Copyright (c) 2001 Network Appliance, Inc.
 * All rights reserved.
 */


/* 
 * Modified Cobion AG
 * 2002-09-21
 */
#include <stdio.h>
#include "iserver.h"
#include "assert.h"
#include "gbuf.h"
#ifdef CbWIN32
#include <string.h>
#include <Windows.h>
#else
#include <strings.h>
#endif

#include "api.h"

#include <ctype.h>
#ifdef CbUNIX
#include<sys/time.h>
#include<unistd.h>
#else
#include <time.h>
#endif


/* GLOBALS */
int modify_control = 0;
int cache_control  = 0;

/* ------------------------------------------------------------------------- */
static char G_ICAP_HTTP_1_0_403_HEADER[] =
	"HTTP/1.0 403 Forbidden\r\n" \
	"Content-Type: text/html\r\n" \
	"Content-Length: %d\r\n" \
	"Pragma: no-cache\r\n" \
	"\r\n";

static char G_ICAP_HTTP_1_1_403_HEADER[] =
	"HTTP/1.1 403 Forbidden\r\n" \
    "Connection: close\r\n" \
	"Content-Length: %d\r\n" \
	"Content-Type: text/html\r\n" \
	"Pragma: no-cache\r\n" \
	"\r\n";

static char G_ICAP_HTTP_1_0_200_HEADER[] =
	"HTTP/1.0 200 OK\r\n" \
	"Content-Type: text/html\r\n" \
	"Content-Length: %d\r\n" \
	"Pragma: no-cache\r\n" \
	"\r\n";

/* ------------------------------------------------------------------------- */
/* callback for content checking, ... */

char* IcapReqModCheckUrl( const char* ip
                        , const char* user
                        , const char* url );

const char* IcapGenerateISTag();

/* ------------------------------------------------------------------------- */
char* icapPrintEncapsResBody(char* strbuf, int offset) 
{
    if(strbuf) {
        sprintf( strbuf
               , "Encapsulated: res-hdr=0, res-body=%d\r\n", offset );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintEncapsReqBody(char* strbuf, int offset) 
{
    if(strbuf) {
        sprintf( strbuf
               , "Encapsulated: req-hdr=0, req-body=%d\r\n", offset );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintEncapsNullBody(char* strbuf, int offset) 
{
    if(strbuf) {
        sprintf( strbuf
               , "Encapsulated: req-hdr=0, null-body=%d\r\n", offset );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintEncapsOptBody(char* strbuf, int optbodylength) 
{
    if(strbuf) {
        if( optbodylength==0 )
            sprintf( strbuf
                   , "Encapsulated: null-body=0\r\n" );
        else
            sprintf( strbuf
                   , "Encapsulated: opt-body=0\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintCacheControl(char* strbuf) 
{
    if(strbuf) {
        sprintf( strbuf
               , "Cache-Control: max-age=%d\r\nAttribute: none\r\n", 360000 );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* 
icapPrintISTag(char* strbuf)
{
    if(strbuf) {
		PTHREAD_MUTEX_LOCK(cinfo.stat_istag_mtx);
		strcpy(strbuf, cinfo.stat_istag);
		PTHREAD_MUTEX_UNLOCK(cinfo.stat_istag_mtx);
		return strbuf + strlen(strbuf);
    } else
        return 0;
}
char* 
icapPrintConnectionClose(char* strbuf)
{
    if(strbuf) {
		strcpy(strbuf, "Connection: close\r\n" );
		return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
static char G_ICAP_RESPONSE_MASK[] = "ICAP/1.0 %d %s\r\n";
char* 
icapPrintResponseHeader(char* strbuf, int status)
{
    if(strbuf) {
        switch(status) {
        case 100:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Continue");
            break;
        case 200:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "OK");
            break;
        case 204:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "No Content");
            break;
        case 400:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Bad Request");
            break;
        case 404:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Not Found");
            break;
        case 405:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Method Not Allowed");
            break;
        case 408:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Request Time-out");
            break;
        case 500:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Internal Server Error");
            break;
        case 501:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Not Implemented");
            break;
        case 502:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Bad Gateway");
            break;
        case 503:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "Service Unavailable");
            break;
        case 505:
            sprintf(strbuf, G_ICAP_RESPONSE_MASK, status, "HTTP Version not supported");
            break;
        default:
            return 0;
        }
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintMethods(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf
               , "Methods: REQMOD, OPTIONS\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapGenerateDate()
{
    // expected format: "Tue, 14. Jan 2003  17:38:33 GMT"
    static const char day_names[7][4] =
      { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
    static const char month_names[12][4] =
      { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

    struct tm *t;
    time_t current_time;
    char* buf;

    buf = (char*)malloc(64*sizeof(char));
    memset( buf, 0, sizeof(buf) );

    time(&current_time); /* get current time */

    /* get and save the gmt */
    {
#ifdef HAVE_GMTIME_R
        struct tm dummy;
        t = gmtime_r(&current_time, &dummy);
#else
        t = gmtime(&current_time);
#endif
    }

    /* Format: "Sun, 06 Nov 1994 08:49:37 GMT" */
    sprintf( buf
            , "%s, %02d %s %4d %02d:%02d:%02d GMT"
            , day_names[t->tm_wday]
            , t->tm_mday
            , month_names[t->tm_mon]
            , t->tm_year + 1900
            , t->tm_hour
            , t->tm_min
            , t->tm_sec );
	return buf;
}
/* ------------------------------------------------------------------------- */
char* icapPrintDate(char* strbuf) 
{
/* "Date: Tue, 14. Jan 2003  17:38:33 GMT\r\n" */
    if(strbuf) {
        char * date;
        date = icapGenerateDate();
        if(date) {
            sprintf( strbuf
                   , "Date: %s\r\n", date  );
            free(date);
        } else
            strcpy( strbuf
                   , "Date: Tue, 14. Jan 2003  17:38:33 GMT\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintService(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf
               , "Service: Proventia Web Filter 2.0\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintServiceID(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf, "Service-ID: pvweb\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintMaxConnections(char* strbuf) 
{
    if(strbuf) {
        sprintf( strbuf, "Max-Connections: %d\r\n", cinfo.opt_num_thr );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintOptionsTTL(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf, "Options-TTL: 3600\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintPreview(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf, "Preview: 4096\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintX(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf, "X-Include: X-Authenticated-User,X-Client-IP\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintTransferPreview(char* strbuf) 
{
    if(strbuf) {
        strcpy( strbuf, "Transfer-Preview: *\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* icapPrintCRLF(char* strbuf)
{
    if(strbuf) {
        strcpy( strbuf, "\r\n" );
        return strbuf + strlen(strbuf);
    } else
        return 0;
}
/* ------------------------------------------------------------------------- */
char* 
icapParseUserProfile(const char *icap_hdr) 
{
    char* pos1, *pos2;
    char* user_profile = 0;
    pos1 = strstr(icap_hdr, "profile=");
    if(pos1) {
        pos1 += strlen("profile=");
        pos2 = strstr(icap_hdr, "ICAP/1.0");
        user_profile = malloc(pos2-pos1);
        memcpy(user_profile, pos1, pos2-pos1);
        user_profile[pos2-pos1-1] = 0;
    }
    return user_profile;
}
/* ------------------------------------------------------------------------- */
char* 
api_options_response(const char* icap_uri)
{
	/* return a pointer to the options response string */
	char *options, *cur_pos;
    options = (char*)malloc(3000*sizeof(char));

    cur_pos = icapPrintResponseHeader(options, 200);
    cur_pos = icapPrintDate(cur_pos);
    cur_pos = icapPrintEncapsOptBody(cur_pos, 0);
    cur_pos = icapPrintISTag(cur_pos);
    cur_pos = icapPrintMaxConnections(cur_pos);
    cur_pos = icapPrintMethods(cur_pos);
    cur_pos = icapPrintOptionsTTL(cur_pos);
    cur_pos = icapPrintPreview(cur_pos);
    cur_pos = icapPrintService(cur_pos);
    cur_pos = icapPrintServiceID(cur_pos);
    cur_pos = icapPrintTransferPreview(cur_pos);
    cur_pos = icapPrintX(cur_pos);   
    cur_pos = icapPrintCRLF(cur_pos);   
    
	if (cinfo.opt_verbose)
		iserver_loginfo("api:Sending options:\n%s", options);
	return options;
}
/* ------------------------------------------------------------------------- */
char*
api_preview_reqmod(const char* icap_hdr, 
		   const char* client_hdr, 
		   const char* post_msg_body,
		   int post_msg_body_len,
		   int preview_size)
{
	char *scratch;
	char str[300], *trans_id_str, *trans_id_str_end;

	char not_interested_header[] = "ICAP/1.0 204 No Content\r\n";
	int msg_size=0, trans_id_len=0, offset;
	char* not_interested_msg;
	
	assert(icap_hdr&&client_hdr&&post_msg_body);
	
    if (cinfo.opt_modify_percent) {
		return NULL;
    } else {
		/* Construct 204 with IS-TAG header and Transaction ID 
		   (if it exists) */
        memset(str, '\0', 300);

        icapPrintISTag(str);

		msg_size = strlen(not_interested_header);
		msg_size += strlen(str);
		
		trans_id_str = (char*) strstr(icap_hdr, "Transaction-Id:");
		
		if (trans_id_str) {
		         trans_id_str_end = (char*) strstr(trans_id_str, "\n");     
			 scratch = trans_id_str;
			 trans_id_str_end++;
		 	 while(scratch != trans_id_str_end) {
			         trans_id_len++;
				 scratch++;
			 }
			 msg_size += trans_id_len;
		}
		
		not_interested_msg = (char*) malloc((msg_size+5)*sizeof(char));
		memset(not_interested_msg, '\0', (msg_size+5)*sizeof(char));
		memcpy(not_interested_msg, 
		 	not_interested_header,   
			strlen(not_interested_header));
		offset = strlen(not_interested_header);
		memcpy(not_interested_msg+offset, str, strlen(str));
		offset += strlen(str);
		memcpy(not_interested_msg+offset, "\r\n", 2);
		offset += 2;
		if (trans_id_len) {
			memcpy(not_interested_msg+offset,
			       trans_id_str,
			       trans_id_len);
			offset += trans_id_len;
		}
		memcpy(not_interested_msg+offset, "\r\n", 2);
		return not_interested_msg;
	}    
}
/* ------------------------------------------------------------------------- */
static int
api_reqmod_cachable()
{
	if (((cache_control++)%100)>=cinfo.opt_reqmod_cache_percent) 
		return 0;
	return 1;  
}
/* ------------------------------------------------------------------------- */
/* for squid mainly, 
	- convert relative path in GET /xxx to absolute path http://host/xxx
	- return complete url for later content checking
*/
int
process_client_hdr( const char* client_hdr, char** new_client_hdr, char** urlpath )
{
	int i, urlpath_len;
	char* posBeg        = (char*)client_hdr;
	char* posEnd        = 0;
    char* httpMethod    = 0;
	char* httpUrl       = 0;
	char* httpVersion   = 0;
	char* httpReqHeader = 0;

    if( client_hdr==0 || urlpath==0 || new_client_hdr==0 ) {
        return -1;      
    }
    (*urlpath) = 0;
    (*new_client_hdr) = 0;

    /* ----- parse ------- */
    posEnd = strstr(posBeg, " ");
    if(posEnd==0)
        return -1;
    
    httpMethod = (char*)malloc(posEnd-posBeg+1);
    memcpy(httpMethod, posBeg, posEnd-posBeg);
    httpMethod[posEnd-posBeg] = 0;

    /* next token */
    posBeg = posEnd; 

    /* skip SP */
    posBeg++;

	posEnd = strstr(posBeg, "\n");
    if(posEnd==0) {
        free(httpMethod);
        return -1;
    }
    httpReqHeader = posEnd+1;
    
    *urlpath = (char*)malloc(posEnd-posBeg);
    memcpy(*urlpath, posBeg, posEnd-posBeg);
    (*urlpath)[posEnd-posBeg-1] = 0;

    for (i=posEnd-posBeg-6; i > 0; i--) {
		if (memcmp((*urlpath)+i, "HTTP/", 5)==0) {
            (*urlpath)[i-1] = 0;
            httpVersion = strdup((*urlpath)+i);
			break;
		}
    }
    if( (*urlpath)[0]!='/'  )
    {
        free(httpVersion);
	    free(httpMethod);
	    return 0;
    }
    /* relative path; construct absolute path */
    /* get host */
    posBeg = strstr(posBeg, "\nHost:");
    if( posBeg==0 ) 
    {
        free(httpVersion);
	    free(httpMethod);
        free(*urlpath);
        *urlpath=0;
        return -1;
    }

    posBeg += 6;
    while(isspace(*posBeg))
        posBeg++;

	posEnd = strstr(posBeg, "\r\n" );
	if( posEnd==0 ) 
    {
        free(httpVersion);
	    free(httpMethod);
        free(*urlpath);
        *urlpath=0;
        return -1;
    }

    /* --- */
    urlpath_len = strlen(*urlpath);
    httpUrl = (char*)malloc((posEnd-posBeg)+urlpath_len+8);
    
    memcpy(httpUrl, "http://", 7);
    memcpy(httpUrl+7, posBeg, posEnd-posBeg); /* Host */
    memcpy(httpUrl+7+(posEnd-posBeg), *urlpath, urlpath_len+1); /* urlpath + \0 */       

    free(*urlpath);
    *urlpath = httpUrl;

    if( cinfo.opt_make_path_absolute )
    {
        urlpath_len = strlen(*urlpath);
        /* --- */
        *new_client_hdr = (char*)malloc( strlen(httpMethod) 
                                       + strlen(httpUrl)
                                       + strlen(httpVersion) 
                                       + strlen(httpReqHeader)
                                       + 5);

	    sprintf( *new_client_hdr, "%s %s %s\r\n%s"
                                  , httpMethod
		                          , httpUrl
 		                          , httpVersion 
		                          , httpReqHeader );
    }
    free(httpVersion);
	free(httpMethod);
    return 0;
}
/* ------------------------------------------------------------------------- */
ca_t* 
api_modify_reqmod( const char* icap_hdr
                 , const char* _client_hdr
                 , const char* _post_msg_body
                 , int post_msg_body_len
                 , const char* user
                 , const char* client_ip )
{
    /* reconstruct message to echo back to client*/
    char icap_response_header[1024];
    char body_size_str[20];

    int res_body_off             = 0;
    int client_header_len        = 0;
    int icap_response_header_len = 0;
    int body_size_str_len        = 0;

	char* client_hdr          = (char*)_client_hdr;
	char* post_msg_body       = (char*)_post_msg_body;
	char* modified_client_hdr = 0;
    char* new_http_hdr = 0;
    char* complete_url        = 0;
	char* check_url_result    = 0;
    char* result              = 0;
    char* cur_pos             = 0;
    char* user_profile        = 0;

    ca_t* res = (ca_t*) malloc(sizeof(ca_t));

	res->beg = 0;
	res->len = 0;
	
	assert(icap_hdr && client_hdr && post_msg_body);
	
	if (cinfo.opt_verbose) {
        iserver_loginfo("incoming icap request");
        iserver_loginfo("%s", icap_hdr);
        if(client_hdr && strcmp("",client_hdr))
            iserver_loginfo("%s", client_hdr);
        else
            iserver_loginfo("client header missed!");
        if(post_msg_body && strcmp("",post_msg_body))
            iserver_loginfo("postmessagebody length: %d", post_msg_body_len);
        else
            iserver_loginfo("no postmessagebody");
	}

    if( cinfo.opt_health_check ) {
        /* Health-Check */
        if((char*)strstr(client_hdr, "Host: icap.health.check\r\n"))
            complete_url = strdup("http://icap.health.check/test.exe");
    }    
    if (complete_url==0) {

	    /* convert client header*/
        if(client_hdr)
	        process_client_hdr(client_hdr, &modified_client_hdr, &complete_url );

	    if( modified_client_hdr ) {
        	if (cinfo.opt_verbose)
		        iserver_loginfo("client header modified");
		    client_hdr = modified_client_hdr;
	    }

        if(cinfo.opt_user_profile)
            user_profile = icapParseUserProfile(icap_hdr);
        /* --------------------------------------------------------------------- */
        /* Obtain the blocking information of a given url.                       */
	    /* if check_url_result != 0, then this contains the blockpage	         */
        if(complete_url!=0) {
            if(user_profile)
    	        check_url_result = IcapReqModCheckUrl( client_ip, user_profile, complete_url );
            else
    	        check_url_result = IcapReqModCheckUrl( client_ip, user, complete_url );
        }

	    if( check_url_result ) 
        {
            /* modify the post message body */
            /* the new body starts after the http header */
		    post_msg_body = strstr(check_url_result, "\r\n\r\n");
		    if( post_msg_body ) 
            {
			    post_msg_body += 4;	/* now points to content */
			    post_msg_body_len = strlen(post_msg_body);

                switch(cinfo.opt_http_header) {
                case ICAP_HTTP_1_1_403_HEADER: 
                	new_http_hdr = malloc(strlen(G_ICAP_HTTP_1_1_403_HEADER)+10);
                	sprintf(new_http_hdr, G_ICAP_HTTP_1_1_403_HEADER, post_msg_body_len );
        		    break;
                case ICAP_HTTP_1_0_200_HEADER: 
					new_http_hdr = malloc(strlen(G_ICAP_HTTP_1_0_200_HEADER)+10);
					sprintf(new_http_hdr, G_ICAP_HTTP_1_0_200_HEADER, post_msg_body_len );
        		    break;
                default: 
					new_http_hdr = malloc(strlen(G_ICAP_HTTP_1_0_403_HEADER)+10);
					sprintf(new_http_hdr, G_ICAP_HTTP_1_0_403_HEADER, post_msg_body_len );
                };
                client_hdr = new_http_hdr;
		    }
			else
				post_msg_body_len= 0;
	    }
    }

    client_header_len = strlen(client_hdr);

    /* Generate Response Header */
    cur_pos = icap_response_header;
    cur_pos = icapPrintResponseHeader(cur_pos, 200);

    /* Generate ISTag */
    if( cinfo.opt_istag_first )
        cur_pos = icapPrintISTag(cur_pos);    
	
	/* Generate Connection: close */
	if( !cinfo.opt_pers_conn )
		cur_pos = icapPrintConnectionClose(cur_pos);

    /* Generate Cache-Control and Attribute header */
    if (api_reqmod_cachable())
        cur_pos = icapPrintCacheControl(cur_pos);
 
    /* Generate Encapsulated header */
	if( check_url_result ) /* Block-Page */
        cur_pos = icapPrintEncapsResBody(cur_pos, client_header_len);
	else if (post_msg_body_len) 
        cur_pos = icapPrintEncapsReqBody(cur_pos, client_header_len);
	else
        cur_pos = icapPrintEncapsNullBody(cur_pos, client_header_len);
			
    /* Generate ISTag */
    if( !cinfo.opt_istag_first )
        cur_pos = icapPrintISTag(cur_pos);

    memcpy(cur_pos, "\r\n", 3); 
    cur_pos +=2;
	icap_response_header_len = (int)(cur_pos-icap_response_header);

    if (post_msg_body_len>0) {
		sprintf(body_size_str, "%X", post_msg_body_len );
		body_size_str_len = strlen(body_size_str);
	}

    res->len = icap_response_header_len
             + client_header_len
             + body_size_str_len
             + post_msg_body_len;

    if (post_msg_body_len>0) { 
        /* 10 = 2 "\r\n" terminate body size str
              + 2 "\r\n" terminate post body
              + 5 "0\r\n\r\n" Last chunk
              + 1 null term. */
		res->len += 10;
	} else if (post_msg_body_len) {
		/* 7 = 5 "0\r\n\r\n" Last chunk
		     + 1 null term. 
             + 1 post_msg_body_len==-1*/
		res->len += 7;
	} else {
		/* 1 = null term. */
		res->len += 1;
	}

	//Let's try it without this, it produces garbage sent to the client
	//res->len+= 256;	// we have a bufferoverwrite somewhere, try for now

    result = (char*) malloc(res->len + 256 );
	memset( result, 0x00, res->len + 256 );

    strcpy(result, icap_response_header);
    strcat(result, client_hdr);

	if (post_msg_body_len>0) {
		if( check_url_result )
		{
			strcat(result, body_size_str);
			strcat(result, "\r\n");
			strcat(result, post_msg_body);
			strcat(result, "\r\n0\r\n\r\n"); 
		}
		else
		{
			char * tmp;
			tmp = result;
			strcat(tmp, body_size_str);
			strcat(tmp, "\r\n");
      		tmp = tmp + strlen(tmp);
			memcpy(tmp, post_msg_body, post_msg_body_len);
     		tmp += post_msg_body_len;
			strcpy(tmp, "\r\n0\r\n\r\n");
//			tmp += strlen("\r\n0\r\n\r\n");
		}
	} else if (post_msg_body_len) {
		strcat(result, "0\r\n\r\n");
	}
	
    if (cinfo.opt_verbose) {
        if(new_http_hdr || modified_client_hdr)
            iserver_loginfo("response with modifications, size: %d (%d)"
            , strlen(result), res->len );
        else
            iserver_loginfo("response without modifications, size: %d (%d)"
            , strlen(result), res->len);
        iserver_loginfo("%s", result);
    }
    res->beg = result;

    if( new_http_hdr )
		free(new_http_hdr);
    if( modified_client_hdr )
		free(modified_client_hdr);
	if( complete_url )
		free(complete_url);
	if (check_url_result)
		free( check_url_result );
	if (user_profile)
		free( user_profile );

	return res;	
}
/* ------------------------------------------------------------------------- */
