// <ISS EXTENSION>

#include "issextension.h"

#ifdef ISS_EXTENSION

#include "SysInclude.h"
#include "SysDep.h"
#include "SvrUtils.h"
#include "StrUtils.h"
#include "SysInclude.h"
#include "SysDep.h"
#include "AppDefines.h"
#include "MessQueue.h"
#include "BuffSock.h"
#include "MiscUtils.h"
#include "SvrDefines.h"
#include "UsrUtils.h"
#include "SMAILUtils.h"

#ifdef ISS_EXTENSION_CONNECT_PVMAIL
#include "clientchannel.h"
#endif

#include <stdio.h>

#ifndef _WIN32
#include <sys/types.h>
#include <utime.h>
#endif

extern bool bServerDebug;
#define ISS_DEBUG_MSG(a) { if (bServerDebug) printf a; }

namespace ISSExtension
{
#ifdef ISS_EXTENSION_STATISTICS
/*
#ifndef _WIN32
inline long InterlockedExchange( long* dest, long val )
{
    long ret;
    __asm__ __volatile__( "lock; xchgl %0,(%1)"
                          : "=r" (ret) :"r" (dest), "0" (val) : "memory" );
    return ret;
}
inline long InterlockedExchangeAdd( long* dest, long incr )
{
    long ret;
    __asm__ __volatile__( "lock; xaddl %0,(%1)"
                          : "=r" (ret) : "r" (dest), "0" (incr) : "memory" );
    return ret;
} 

inline long InterlockedIncrement(long* pCount)
{
    return InterlockedExchangeAdd( pCount, 1 ) + 1;
}

inline long InterlockedDecrement(long* pCount)
{
    return InterlockedExchangeAdd( pCount, -1 ) - 1;
}
#endif
*/
namespace ISSQueueStatisticalData
{

// JW: CR 502367 - secure IPC communication
// static bool S_dirty= false;

class Counter
{
public:
	Counter(const char* name)
		: _name(SysStrDup(name))
		, _value(0)
	{
	}
	~Counter()
	{
		SysFree(_name);
	}
	size_t GetValue() const
	{
		return _value;
	}
	const char* GetName() const
	{
		return _name;
	}

	void Set(size_t count)
	{
//		S_dirty= true;
		size_t newval= count;
		InterlockedExchange(reinterpret_cast<long*>(&_value), count);
	}
	Counter & operator++()
	{
//		S_dirty= true;
		_value= InterlockedIncrement(reinterpret_cast<long*>(&_value));
		return *this;
	}
	Counter & operator--()
	{
//		S_dirty= true;
		_value= InterlockedDecrement(reinterpret_cast<long*>(&_value));
		return *this;
	}
private:
	char* _name;
	size_t _value;
};

#ifdef _DEBUG
static void DebugCounter(const char* what, const Counter & value)
{
#ifdef _WIN32
#define FILENAME "C:\\iss_counter_values.txt"
#else
#define FILENAME "/tmp/iss_counter_values.txt"
#endif
	FILE * outfile= fopen(FILENAME, "a");
	if (outfile)
	{
		static bool first= true;
		if (first)
		{
			first= false;
		}
		time_t t= time(0);
		char ctimebuff[30]= "";
		strcpy(ctimebuff, ctime(&t));
		char* pos= strchr(ctimebuff, '\n');
		if (pos)
			*pos= 0;
		fprintf(outfile, "%s;%s;%s;%d\n", ctimebuff, value.GetName(), what, value.GetValue());
		fclose(outfile);
	}
}

#define DEBUG_COUNTER(a, c)	DebugCounter(a, c)

#else
#define DEBUG_COUNTER(a, c)	0
#endif


class ProventiaCommThread
{
	static ProventiaCommThread* S_thread;
	time_t _lastSent;
	static SYS_MUTEX S_mutex;
	static bool S_stop;

	SYS_THREAD _thread;

	void AppendCounter(const char* countername, long countervalue, DynString & buffer);

public:
	ProventiaCommThread();
	
	~ProventiaCommThread();

	static ProventiaCommThread* TheThread();
	
	static void Exit();

	static unsigned int ThreadProc(void *pThreadData);

	void SendToProventia();

	void Start();
};


#define IMPLEMENT_COUNTER(a) \
	static Counter S_##a##Count(#a); \
	void ISSExtension::ISSQueueStatisticalData::Incr##a() { ++S_##a##Count; DEBUG_COUNTER("incremented", S_##a##Count); } \
	void ISSExtension::ISSQueueStatisticalData::Decr##a() { --S_##a##Count; DEBUG_COUNTER("decremented", S_##a##Count); } 

IMPLEMENT_COUNTER(Sent)
IMPLEMENT_COUNTER(Resend)
IMPLEMENT_COUNTER(Send)

ProventiaCommThread* ProventiaCommThread::S_thread= 0;
SYS_MUTEX ProventiaCommThread::S_mutex= SysCreateMutex();
bool ProventiaCommThread::S_stop= false;

ProventiaCommThread::ProventiaCommThread()
: _lastSent(time(0))
, _thread(0)
{
	SysLogMessage(LOG_LEV_MESSAGE, "ISS statistical data communication thread created\n");
}

ProventiaCommThread::~ProventiaCommThread()
{
	SysCloseThread(_thread, 1);
	SysCloseMutex(S_mutex);
	SysLogMessage(LOG_LEV_MESSAGE, "ISS statistical data communication thread terminated\n");
}

void ProventiaCommThread::Start()
{
	if (!_thread)
	{
		_thread= SysCreateThread(ThreadProc, this);
		SysLogMessage(LOG_LEV_MESSAGE, "ISS statistical data communication thread started\n");
	}
}

static void CheckTimeZone()
{
	static const char clockfile[]= 
#ifdef __LINUX__
		"/etc/sysconfig/clock";
#else
		 "clock";
#endif

	static char last_timezone[1024]= "";
	static time_t last_mod_time= 0;
	static size_t call_count= 0;

	if (call_count==0 || call_count>=10)
	{
		if (call_count >= 10)
			call_count= 0;
		struct stat s;
		if (stat(clockfile, &s) == 0)
		{
			if (s.st_mtime != last_mod_time)
			{
				FILE * fp= fopen(clockfile, "r");
				if (fp)
				{
					char buffer[1024]= "";
					while (fgets(buffer, sizeof(buffer), fp))
					{
						// e.g. ZONE="Europe/Berlin"
						if (strncmp(buffer, "ZONE=", 5)==0)
						{
							char temp[1024]= "";
							char* tt= strchr(buffer, '"');
							if (tt)
								strncpy(temp, tt+1, sizeof(temp)-1);
							else
								strncpy(temp, buffer+5, sizeof(temp)-1);
							tt= strchr(temp, '"');
							if (tt)
								*tt= 0;
							tt= strchr(temp, '\r');
							if (tt)
								*tt= 0;
							tt= strchr(temp, '\n');
							if (tt)
								*tt= 0;
							if (strcmp(temp, last_timezone))
							{
#ifdef __LINUX__
								char ttt[1024]= "";
								snprintf(ttt, sizeof(ttt), ":/usr/share/zoneinfo/%s", temp);
								setenv("TZ", ttt, 1);
#else
								char* ttt= 0;
								STRSPRINTF(ttt, "TZ=%s", temp);
								if (ttt)
								{
									putenv(ttt);
									SysFree(ttt);
								}
#endif
								tzset();
								strncpy(last_timezone, temp, sizeof(last_timezone)-1);
#ifdef __LINUX__
								SysLogMessage(LOG_LEV_MESSAGE, "Timezone set to '%s'", temp);
#endif
							}
							break;
						}
					}
					fclose(fp);
				}
				else
				{
#ifdef __LINUX__
					SysLogMessage(LOG_LEV_ERROR, "Error opening file '%s'", clockfile);
#endif
				}
				last_mod_time= s.st_mtime;
			}
		}
		else
		{
#ifdef __LINUX__
			SysLogMessage(LOG_LEV_ERROR, "Error stating file '%s'", clockfile);
#endif
		}
	}
	++call_count;
}

void ProventiaCommThread::SendToProventia()
{
// JW: CR 502367 - secure IPC communication
//	if (!S_dirty)
//		return;

	CheckTimeZone();
	time_t now= time(0);
	if ((now - _lastSent)>=60)	// every minute
	{
		MutexAutoPtr locker(S_mutex);
		try
		{
			_lastSent= now;
			DynString data, result;
			StrDynInit(&data);
			StrDynInit(&result);
			StrDynAdd(&data, "smtp-stats\t", -1);
//			S_dirty= false;
			size_t sent= S_SentCount.GetValue();
			size_t send= S_SendCount.GetValue();
			size_t resend= S_ResendCount.GetValue();
//			size_t frozen= S_FrozenCount.GetValue();
//			size_t unchecked= S_UncheckedCount.GetValue();
			AppendCounter(S_SentCount.GetName(), sent, data);
			AppendCounter(S_SendCount.GetName(), send, data);
			AppendCounter(S_ResendCount.GetName(), resend, data);
//			AppendCounter(S_FrozenCount.GetName(), frozen, data);
//			AppendCounter(S_UncheckedCount.GetName(), unchecked, data);
			size_t timeout= SvrGetConfigInt("ISSIPCSendTimeout", ISS_IPC_SEND_TIMEOUT, INVALID_SVRCFG_HANDLE);
			if (SendIPCCommand(data, result, timeout))
//			if (ipc.Send(StrDynGet(&data), StrDynSize(&data), result, ISS_IPC_SEND_TIMEOUT))
			{
				size_t temp;
				for (temp= 0; temp<sent; ++temp)
					--S_SentCount;
//				for (temp= 0; temp<frozen; ++temp)
//					--S_FrozenCount;
//				for (temp= 0; temp<unchecked; ++temp)
//					--S_UncheckedCount;
			}
			StrDynFree(&data);
			StrDynFree(&result);
		}
		catch(...)
		{}
	}
	else if (_lastSent > now)
		_lastSent= now;
}

// layout of counter
//	NAME;0123456789;
//  ;   ;VALUE;

void ProventiaCommThread::AppendCounter(const char* countername, long countervalue, DynString & buffer)
{
	if (StrDynSize(&buffer)>0)
		StrDynAdd(&buffer, "\t", 1);
	StrDynAdd(&buffer, countername, -1);
	StrDynAdd(&buffer, "\t", 1);

	char *temp= StrSprint("%d", countervalue);
	StrDynAdd(&buffer, temp, -1);
	SysFree(temp);
}


ProventiaCommThread* ProventiaCommThread::TheThread()
{
	if (!S_thread)
	{
		MutexAutoPtr locker(S_mutex);
		if (!S_thread)
		{
			try
			{
				S_thread= new ProventiaCommThread;
				S_thread->Start();
			}
			catch(...)
			{}
		}
	}
	return S_thread;
}

void ProventiaCommThread::Exit()
{
	SysLogMessage(LOG_LEV_MESSAGE, "Exiting ProventiaCommThread");
	MutexAutoPtr locker(S_mutex);
	try
	{
		if (S_thread)
		{
			SysMsSleep(2000);
			ProventiaCommThread * temp= S_thread;
			S_thread= 0;
			delete temp;
			SysLogMessage(LOG_LEV_MESSAGE, "ProventiaCommThread deleted");
		}
	}		
	catch(...)
	{
	}
}

unsigned int ProventiaCommThread::ThreadProc(void * data)
{
	SysLogMessage(LOG_LEV_MESSAGE, "Entering ProventiaCommThread::ThreadProc");
	while (S_thread)
	{
		S_thread->SendToProventia();
		SysMsSleep(2000);
	}
	SysLogMessage(LOG_LEV_MESSAGE, "Leaving ProventiaCommThread::ThreadProc");
	return 0;
}


}	// namespace	ISSQueueStatisticalData
#endif	// ISS_EXTENSION_STATISTICS


#undef	ISS_DEBUG_MSG


// custom greeting message
char* GetSmtpGreetingMessage(SVRCFG_HANDLE cfg, char * msg, size_t buffsize)
{
	if (!msg || !buffsize)
		return 0;
	*msg= 0;
	// replace SMTP_SERVER_NAME in greeting and error message
	// read config string "SmtpSvrGreetingMessage"
	bool releaseConfig= false;
	if (cfg==INVALID_SVRCFG_HANDLE)
	{
		cfg= SvrGetConfigHandle();
		if (cfg == INVALID_SVRCFG_HANDLE)
			return msg;
		releaseConfig= true;
	}
	char * retval= SvrGetConfigVar(cfg, "SmtpSvrGreetingMessage");
	if (retval)
	{
		strncpy(msg, retval, buffsize-1);
		msg[buffsize-1]= 0;
		SysFree(retval);
	}
	if (releaseConfig)
		SvrReleaseConfigHandle(cfg);
	return msg;
}

#ifdef ISS_EXTENSION_CONNECT_PVMAIL
bool SendIPCCommand(DynString & command, DynString & result, size_t timeout)
{
	return SendIPCCommand(StrDynGet(&command), StrDynSize(&command), result, timeout);
}


bool SendIPCCommand(const char* command, size_t len, DynString & result, size_t timeout)
{
	if (timeout == 0)
		timeout= ISS_IPC_SEND_TIMEOUT;
	ISSExtension::IPC ipc;

	time_t t= time(0);
	t+= timeout/1000;
	char* buffer= StrSprint("timeout=%ld\t%s", t, command);
	bool res= ipc.Send(buffer, strlen(buffer)+1, result, timeout);
	SysFree(buffer);

	return res;
}


bool SendMsgTrackingCommand(const char* command, size_t len, DynString & result, size_t timeout)
{
	return SendIPCCommand(command, len, result, timeout);
}

void InitIPC()
{
	ISSQueueStatisticalData::ProventiaCommThread::TheThread();
}

void DeinitIPC()
{
	ISSQueueStatisticalData::ProventiaCommThread::Exit();
}

#endif


}	// namespace ISSExtension
#endif	// ISS_EXTENSION
// </ISS EXTENSION>


