// <ISS EXTENSION>

#ifdef ISS_EXTENSION_FILTER


#define ISS_DEBUG_MSG(a) { MessageFilter::Log a ; }

namespace ISSExtension
{
class MessageFilter
{
	static SYS_MUTEX S_mutex;
	static FILE * S_LogFile;

public:
	static int FilterMessage(SMTPSession & SMTPS, char * & pszError);

	static void Log(const char* format, ...);

	static void Cleanup();

	static void Shutdown();

private:
	static void UpdateQueues(SMTPSession & SMTPS);

	static void BuildQueueFromFile(const char* path, const char* filename);

	static bool FindInQueues(const char* recipient);

	static void OpenLogFile();

	struct FilterQueue
	{
		// file the queue was built from
		char _fileName[SYS_MAX_PATH];
		// last modification time of the file at the time of building the queue
		time_t _timeStamp;
		// array of email addresses - sorted ascencing - for quick binary search
		char ** _addresses;
		// number of addressed in _addresses
		size_t _addressCount;
		// flag if found by current scan
		bool _isHit;

		FilterQueue()
			: _timeStamp(0)
			, _addresses(0)
			, _addressCount(0)
			, _isHit(false)
		{
			_fileName[0]= 0;
		}

		~FilterQueue()
		{
			Cleanup();
		}

		void Cleanup();

		void UpdateFromFile(const char* path);

		static int compare_qsort(const char** left, const char** right)
		{
			return stricmp(*left, *right);
		}
		static int compare_bsearch(const char* left, const char** right)
		{
			return stricmp(left, *right);
		}

		bool Find(const char* recipient);

	private:
		FilterQueue(const FilterQueue & that);
		const FilterQueue & operator=(const FilterQueue & that);
	};

	static FilterQueue ** _queues;
	static size_t _queueCount;
	static size_t _lastUpdate;
};

SYS_MUTEX MessageFilter::S_mutex= SysCreateMutex();
FILE * MessageFilter::S_LogFile= 0;
MessageFilter::FilterQueue ** MessageFilter::_queues= 0;
size_t MessageFilter::_queueCount= 0;
size_t MessageFilter::_lastUpdate= 0;

void MessageFilter::Cleanup()
{
	if (_queues)
	{
		for (size_t i= 0; i<_queueCount; ++i)
			delete _queues[i];
		delete [] _queues;
		_queues= 0;
	}
	_queueCount= 0;
}

void MessageFilter::Log(const char* format, ...)
{
	char *msg = NULL;

	try
	{
		STRSPRINTF(msg, format, format);
		if (!msg)
			return;

		if (bServerDebug)
			printf("ISSFilter: %s", msg);

		if (S_LogFile)
		{
			fprintf(S_LogFile, msg);
			fflush(S_LogFile);
			unsigned long pos= ftell(S_LogFile);
			if (pos > 5 * 1024 * 1024)
			{
				OpenLogFile();
			}
		}
	}
	catch(...)
	{
	}
	if (msg)
		SysFree(msg);
}

bool MessageFilter::FilterQueue::Find(const char* recipient)
{
	if (!_addresses || _addressCount==0)
		return false;

	// plain email text search
	if (0 != bsearch(recipient, _addresses, _addressCount, sizeof(_addresses[0]), 
			(int (*)(const void*, const void*))compare_bsearch))
		return true;

	// supported wildcard:
	//		*@your.domain
	// so the only supported wildcard is the placeholder * for the address part, no
	//	wildcard in the domain part, no *.weber@cobion.com, no nothing, only *@your.domain

	// get the domain part of the recipient
	const char * recipientdomain= strchr(recipient, '@');
	if (!recipientdomain)
		return false;
	++recipientdomain;
	if (*recipientdomain==0)
		return false;
	for (size_t i= 0; i<_addressCount; ++i)
	{
		const char* temp= _addresses[i];
		if (tolower(*temp)>'*')
			break;
		if (*temp == '*' && *(temp+1)=='@')	// ok, is *@your.domain
		{
			temp+= 2;	// point at domain part

			// compare domain parts, if equal it is to be accepted
			if (stricmp(temp, recipientdomain)==0)
				return true;
		}
	}
	return false;
}


void MessageFilter::FilterQueue::UpdateFromFile(const char* path)
{
	SYS_FILE_INFO fi;
	SysGetFileInfo(path, fi);
	if (fi.tMod == _timeStamp)
		return;

	Cleanup();
	_timeStamp= fi.tMod;

	// read all lines
	FILE* fp= fopen(path, "r");
	if (fp)
	{
		char line[1024];
		const size_t CHUNCK_SIZE= 16;
		size_t alloc_size= 0;
		while (MscGetString(fp, line, sizeof(line)-1))
		{
			StrLTrim(line, " \t<\"");
			StrRTrim(line, " \t>\"");

			if (IsEmptyString(line))	// skip empty lines
				continue;

			if (_addressCount >= alloc_size)
			{
				// reallocate and copy
				char ** temp= new char*[alloc_size+= CHUNCK_SIZE];
				memset(temp, 0, alloc_size * sizeof(char*));
				if (_addressCount > 0)
				{
					memcpy(temp, _addresses, _addressCount * sizeof(char*));
					delete [] _addresses;
				}
				_addresses= temp;
			}
			size_t len= strlen(line);
			_addresses[_addressCount]= new char[len+1];
			strcpy(_addresses[_addressCount], line);
			// ISS_DEBUG_MSG(("init\tadded\t%s\n", _addresses[_addressCount]))
			++_addressCount;
		}
		fclose(fp);

		if (_addresses) // sort
		{
			qsort(_addresses, _addressCount, sizeof(char*), 
				(int (*)(const void*, const void*))compare_qsort);
			for (size_t i= 0; i<_addressCount; ++i)
			{
				ISS_DEBUG_MSG(("init\tadded\t%s\n", _addresses[i]))
			}
		}
		ISS_DEBUG_MSG(("inittotal\t%d\t%s\n", _addressCount, path));
	}
	else
		Cleanup();
}

void MessageFilter::Shutdown()
{
	{
		MutexAutoPtr locker(S_mutex);
		Cleanup();
		ISS_DEBUG_MSG(("shutdown\n"));
		if (S_LogFile)
		{
			fclose(S_LogFile);
			S_LogFile= 0;
		}
	}
	SysCloseMutex(S_mutex);
	S_mutex= 0;
	SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter stopped\n");
}


void MessageFilter::FilterQueue::Cleanup()
{
	if (_addresses)
	{
		for (size_t i= 0; i<_addressCount; ++i)
			delete[] (_addresses[i]);
		delete[] (_addresses);
		_addresses= 0;
	}
	_addressCount= 0;
	_timeStamp= 0;
}

bool MessageFilter::FindInQueues(const char* recipient)
{
	// if no queues -> do not filter!!!
	if (!_queues)
		return true;

	size_t callCount= 0;
	for (size_t i= 0; i<_queueCount; ++i)
	{
		if (_queues[i])
		{
			++callCount;
			if (_queues[i]->Find(recipient))
				return true;
		}
	}
	
	// if no queues -> do not filter
	if (callCount == 0)
	{
		Cleanup();
		return true;
	}
	return false;
}

void MessageFilter::BuildQueueFromFile(const char* path, const char* filename)
{
	char fpath[SYS_MAX_PATH]= "";
	sprintf(fpath, "%s%s%s", path, SYS_SLASH_STR, filename);

	// find filename in _queues
	FilterQueue * curr= 0;
	size_t i;
	size_t currPos= 0;
	for (i= 0; i<_queueCount && !curr; ++i)
	{
		if (_queues[i])
		{
			if (stricmp(_queues[i]->_fileName, filename) == 0)
			{
				// found
				curr= _queues[currPos= i];
			}
		}
	}
	bool existed= false;
	if (curr)
	{
		existed= true;
		curr->_isHit= true;
	}
	else
	{
		curr= new FilterQueue;
		curr->_isHit= true;
		strcpy(curr->_fileName, filename);
	}
	curr->UpdateFromFile(fpath);
	
	// empty files are ignored!
	if (curr->_addressCount == 0)
	{
		delete curr;
		if (existed)
			_queues[currPos]= 0;

		return;
	}
	if (!existed)
	{
		// insert into _queues
		// search for first empty entry 
		for (i= 0; i<_queueCount; ++i)
		{
			if (_queues[i]==0)
			{
				_queues[i]= curr;
				break;
			}
		}
		// if there was no empty space, append
		if (i == _queueCount)
		{
			// reallocate and copy
			FilterQueue** temp= new FilterQueue*[_queueCount + 1];
			if (_queueCount)
			{
				memcpy(temp, _queues, _queueCount * sizeof(_queues[0]));
				delete[] _queues;
			}
			_queues= temp;
			_queues[_queueCount++]= curr;
		}
	}
}

void MessageFilter::UpdateQueues(SMTPSession & SMTPS)
{
	time_t now= time(0);
	if (_lastUpdate != 0 && now - _lastUpdate < 60 * 5)	// parse every five minutes
		return;

	_lastUpdate= now;

	bool enablemsgfilter= false;

	char * enabled= SvrGetConfigVar(SMTPS.hSvrConfig, "EnableSmtpISSFilter");
	if (enabled)
	{
		if (strcmp(enabled, "1")==0 || stricmp(enabled, "true")==0 || stricmp(enabled, "yes")==0)
			enablemsgfilter= true;
		SysFree(enabled);
	}
	if (!enablemsgfilter)
	{
		Cleanup();
		SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter disabled\n");
		return;
	}

	if (_queues)
	{
		for (size_t i= 0; i<_queueCount; ++i)
			if (_queues[i])
				_queues[i]->_isHit= false;
	}

	// parse directory MAIL_ROOT/issfilter for "*.allowed" files
	char path[SYS_MAX_PATH]= "";
	char updatefile[SYS_MAX_PATH]= "";
	char filename[SYS_MAX_PATH]= "";
	CfgGetRootPath(path, sizeof(path));
	StrNCat(path, "issfilter", sizeof(path));
	// PVMAIL creates a .update file in MAIL_ROOT/issfilter to signal that queues should be updated
	static bool first= true;
	strcpy(updatefile, path);
	AppendSlash(updatefile);
	StrNCat(updatefile, ".update", sizeof(updatefile));
	if (SysExistFile(updatefile) == 0 && !first)
		return;

	first= false;
	SYS_HANDLE fh= SysFirstFile(path, filename);
	if (fh != SYS_INVALID_HANDLE)
	{
		do
		{
			if (SysIsDirectory(fh) || IsDotFilename(filename))
				continue;

			// check for ".allowed" extension
			const char * temp= strstr(filename, ".allowed");
			if (temp)
			{
				size_t len= strlen(filename);
				if (temp == filename + len - 8)
				{
					// found one!
					BuildQueueFromFile(path, filename);
				}
			}
		} while (SysNextFile(fh, filename));
		SysFindClose(fh);
	}
	SysRemove(updatefile);

	// cleanup all queues that where not found 
	if (_queues)
	{
		size_t deleteCount= 0;
		for (size_t i= 0; i<_queueCount; ++i)
		{
			if (_queues[i])
			{
				if(_queues[i]->_isHit == false)
				{
					delete _queues[i];
					_queues[i]= 0;
					++deleteCount;
				}
			}
			else
				++deleteCount;
		}
		// nothing left, cleanup
		if (_queueCount == deleteCount)
			Cleanup();
	}
}

void MessageFilter::OpenLogFile()
{
	if (S_LogFile)
	{
		fclose(S_LogFile);
		S_LogFile= 0;
	}
	char logfilepath[SYS_MAX_PATH];
	MscLogFilePath("issfilter", logfilepath);
	if (SysExistFile(logfilepath))
	{
		for (int i= 0; i<1000; ++i)
		{
			char backuppath[SYS_MAX_PATH];
			sprintf(backuppath, "%s.bak.%d", logfilepath, i);
			if (!SysExistFile(backuppath))
			{
				SysMoveFile(logfilepath, backuppath);
				break;
			}
		}
	}
	S_LogFile= fopen(logfilepath, "a");
}

// read the spool file and check for every recipient if it is in the allowed lists
// -) if at least one recipient is allowed, continue normal processing
// -) if no recipient is allowed, return error
int MessageFilter::FilterMessage(SMTPSession & SMTPS, char * & pszError)
{
	MutexAutoPtr locker(S_mutex);

	static int first= true;
	if (first)
	{
		OpenLogFile();
		SysLogMessage(LOG_LEV_MESSAGE, "ISSFilter starting\n");
		first= false;
	}

	UpdateQueues(SMTPS);

	if (!_queues)
		return 0;

	ISS_DEBUG_MSG(("checking\t%s (%s)\n", SMTPS.pszRcpt, 
			SMTPS.iRcptCount>1 ? "and others" : "single recipient"));

	// special case: only one recipient, fast handling
	bool ok= false;
	if (SMTPS.iRcptCount == 1)
		ok= FindInQueues(SMTPS.pszRcpt);
	else
	{
	    char line[MAX_SPOOL_LINE] = "";

		unsigned long pos= ftell(SMTPS.pMsgFile);
		rewind(SMTPS.pMsgFile);
		size_t clen= strlen(RCPT_TO_STR);
		bool foundOne= false;
		while (ok==false && MscGetString(SMTPS.pMsgFile, line, sizeof(line)-1))
		{
			// read the message file
			if (strnicmp(line, RCPT_TO_STR, clen) != 0)
				if (foundOne)
					break;
				else
					continue;

			foundOne= true;
			SMTPTrimRcptLine(line);
			char* temp= line+clen;
			temp= StrLTrim(temp, " \t<");
			StrRTrim(temp, ">");
			if (FindInQueues(temp))
			{
				ok= true;
				ISS_DEBUG_MSG(("allowed\t%s\n", temp));
			}
			else
			{
				ISS_DEBUG_MSG(("not allowed, continuing\t%s\n", temp));
			}
		}
		fseek(SMTPS.pMsgFile, pos, SEEK_SET);
	}
	if (ok == false)
	{

		// create custom error message, if empty, XMAIL will return "554 Transaction failed"
		// configure this error message in server.tab
		pszError = SvrGetConfigVar(SMTPS.hSvrConfig, "SmtpISSFilterRejectionMessage");

		ISS_DEBUG_MSG(("mail rejected\t%s (%s)\t%s\n", SMTPS.pszRcpt, 
			SMTPS.iRcptCount>1 ? "and others" : "single recipient", pszError ? pszError : "(XMAIL's default)"));

		ErrSetErrorCode(ERR_FILTERED_MESSAGE);
		return (ERR_FILTERED_MESSAGE);
	}
	else
	{
		ISS_DEBUG_MSG(("mail allowed\t%s (%s)\n", SMTPS.pszRcpt, 
				SMTPS.iRcptCount>1 ? "and others" : "single recipient"));
	}
	return 0;
}

#undef ISS_DEBUG_MSG // { if (bServerDebug) printf a; }

}	// namespace ISSExtension
#endif	// ISS_EXTENSION
// </ISS EXTENSION>


