// <ISS EXTENSION>

#include "issextension.h"

#ifdef ISS_EXTENSION

#include "SysInclude.h"
#include "SysDep.h"
#include "SvrUtils.h"
#include "SysInclude.h"
#include "SysDep.h"
#include "AppDefines.h"
#include "MessQueue.h"
#include "BuffSock.h"
#include "MiscUtils.h"
#include "SvrDefines.h"
#include "UsrUtils.h"
#include "SMAILUtils.h"

#ifndef _WIN32
#include <sys/types.h>
#include <utime.h>
#endif

#define ISS_DEBUG_MSG(a) { if (bServerDebug) printf a; }

namespace ISSExtension
{
#ifdef ISS_EXTENSION_STATISTICS
namespace ISSQueueStatisticalData
{

static bool S_dirty= false;

class Counter
{
public:
	Counter(const char* name)
		: _name(SysStrDup(name))
		, _value(0)
	{
	}
	~Counter()
	{
		SysFree(_name);
	}
	size_t GetValue() const
	{
		return _value;
	}
	const char* GetName() const
	{
		return _name;
	}

	Counter & operator++()
	{
		S_dirty= true;
	#ifdef _WIN32
		_value= InterlockedIncrement(reinterpret_cast<long*>(&_value));
	#else
		_value= osl_incrementInterlockedCount(&_value);
	#endif
		ISS_DEBUG_MSG (("ISS statistical data %s incremented, new value is %d\n", _name, _value));
		return *this;
	}
	Counter & operator--()
	{
		S_dirty= true;
	#ifdef _WIN32
		_value= InterlockedDecrement(reinterpret_cast<long*>(&_value));
	#else
		_value= osl_decrementInterlockedCount(&_value);
	#endif
		ISS_DEBUG_MSG(("ISS statistical data %s decremented, new value is %d\n", _name, _value));
		return *this;
	}
private:
	char* _name;
	size_t _value;
};

#ifdef _DEBUG
static void DebugCounter(const char* file, const char* what, const Counter & value)
{
#ifdef _WIN32
#define FILENAME "C:\\iss_counter_values.txt"
#else
#define FILENAME "/tmp/iss_counter_values.txt"
#endif
	FILE * outfile= fopen(FILENAME, "a");
	if (outfile)
	{
		static bool first= true;
		if (first)
		{
			first= false;
			fprintf(stderr, "see "FILENAME" for ';'-separated xmail queue counter history\n");
			fprintf(outfile, "-----------------------------------------------------------\n");
		}
		time_t t= time(0);
		char ctimebuff[30]= "";
		strcpy(ctimebuff, ctime(&t));
		char* pos= strchr(ctimebuff, '\n');
		if (pos)
			*pos= 0;
		fprintf(outfile, "%s;%s;%s;%s;%d\n", ctimebuff, file, value.GetName(), what, value.GetValue());
		fclose(outfile);
	}
	else
		fprintf(stderr, "ISSCounter %s %s %s %d", file, value.GetName(), what, value.GetValue());
}

#define DEBUG_COUNTER(f, a, c)	DebugCounter(f, a, c)

#else
#define DEBUG_COUNTER(f, a, c)	0
#endif


class ProventiaCommThread
{
	static ProventiaCommThread* S_thread;
	time_t _lastSent;
	static SYS_MUTEX S_mutex;
	static bool S_stop;
	static char S_ExchangeFile[SYS_MAX_PATH];

	SYS_THREAD _thread;

	void AppendCounter(const Counter & counter, char* & buffer, size_t & startpos, size_t & size);
public:
	ProventiaCommThread();
	
	~ProventiaCommThread();

	static ProventiaCommThread* TheThread();
	
	static void Exit();

	static unsigned int ThreadProc(void *pThreadData);

	void SendToProventia();

};

#define IMPLEMENT_COUNTER(a) \
	static Counter S_##a##Count(#a); \
	void ISSExtension::ISSQueueStatisticalData::Incr##a(const char* f) { ++S_##a##Count; ProventiaCommThread::TheThread(); DEBUG_COUNTER(f, "incremented", S_##a##Count); } \
	void ISSExtension::ISSQueueStatisticalData::Decr##a(const char* f) { --S_##a##Count; ProventiaCommThread::TheThread(); DEBUG_COUNTER(f, "decremented", S_##a##Count); } 

IMPLEMENT_COUNTER(Sent)
IMPLEMENT_COUNTER(ReadyToSend)
IMPLEMENT_COUNTER(Frozen)
IMPLEMENT_COUNTER(Unchecked)
IMPLEMENT_COUNTER(Resend)


ProventiaCommThread* ProventiaCommThread::S_thread= 0;
SYS_MUTEX ProventiaCommThread::S_mutex= SysCreateMutex();
bool ProventiaCommThread::S_stop= false;
char ProventiaCommThread::S_ExchangeFile[SYS_MAX_PATH]= "";

ProventiaCommThread::ProventiaCommThread()
: _lastSent(time(0))
{
	_thread= SysCreateThread(ThreadProc, this);
	SysLogMessage(LOG_LEV_MESSAGE, "ISS statistical data communication thread started\n");
}

// SYS_THREAD      SysCreateThread(unsigned int (*pThreadProc) (void *), void *pThreadData)
ProventiaCommThread::~ProventiaCommThread()
{
	SysCloseThread(_thread, 1);
	SysCloseMutex(S_mutex);
	SysLogMessage(LOG_LEV_MESSAGE, "ISS statistical data communication thread terminated\n");
}

void ProventiaCommThread::SendToProventia()
{
	if (!S_dirty)
		return;

	time_t now= time(0);
	if ((now - _lastSent)>=60)	// every minute
	{
		if (!*S_ExchangeFile)
			return;

		MutexAutoPtr locker(S_mutex);

		try
		{
			_lastSent= now;

			// send counter values to proventia
			char * buffer= 0;
			size_t size= 0;
			size_t startpos= 0;
			AppendCounter(S_ResendCount, buffer, startpos, size);
			AppendCounter(S_SentCount, buffer, startpos, size);
			AppendCounter(S_FrozenCount, buffer, startpos, size);
			AppendCounter(S_ReadyToSendCount, buffer, startpos, size);
			AppendCounter(S_UncheckedCount, buffer, startpos, size);

			// TODO: save to socket, file, shared memory ... ?
			FILE * file= fopen(S_ExchangeFile, "wb");
			if (file)
			{
				fwrite(buffer, 1, startpos, file);
				fclose(file);
				S_dirty= false;
				ISS_DEBUG_MSG(("ISS saved statistical data to file %s\n", S_ExchangeFile));
			}
			else
				SysLogMessage(LOG_LEV_WARNING, "ISS could not save statistical data to file %s\n", S_ExchangeFile);				
		}
		catch(...)
		{}
	}
}

// layout of counter
//	NAME;0123456789;
//  ;   ;VALUE;
void ProventiaCommThread::AppendCounter(const Counter & counter, char* & buffer, size_t & startpos,
										size_t & size)
{
	size_t incr= 256;

	size_t len= strlen(counter.GetName()) + 1;	// name + ';'
	len+= 10 + 1;	// value + ';'

	if (len > incr)
		incr= len;

	if (!buffer)
	{
		buffer= (char*)SysAlloc(size= incr);
		*buffer= 0;
	}
	else if ((startpos + len) > size)
		buffer= (char*)SysRealloc(buffer, size+= incr);

	sprintf(buffer+startpos, "%s;%010d;", counter.GetName(), counter.GetValue());
	startpos= strlen(buffer);
}


ProventiaCommThread* ProventiaCommThread::TheThread()
{
	if (!S_thread)
	{
		MutexAutoPtr locker(S_mutex);
		if (!S_thread)
		{
			try
			{
				S_thread= new ProventiaCommThread;
			}
			catch(...)
			{}
		}
	}
	return S_thread;
}

void ProventiaCommThread::Exit()
{
	MutexAutoPtr locker(S_mutex);
	try
	{
		if (S_thread)
		{
			SysMsSleep(2000);
			ProventiaCommThread * temp= S_thread;
			S_thread= 0;
			delete temp;
		}
	}		
	catch(...)
	{
	}
}

unsigned int ProventiaCommThread::ThreadProc(void * data)
{
	// scan the unchecked directory at startup
	SVRCFG_HANDLE hSvrConfig = SvrGetConfigHandle();
	char *dropdir= SvrGetConfigVar(hSvrConfig, "SmtpSvrUncheckedDir");
	if (dropdir && *dropdir)
	{
		char szSpoolPath[SYS_MAX_PATH]= "", filename[SYS_MAX_PATH]= "";
		SvrGetSpoolDir(szSpoolPath, sizeof(szSpoolPath) );
		AppendSlash(szSpoolPath);
		StrNCat(szSpoolPath, dropdir, sizeof(szSpoolPath));

		SYS_HANDLE fh= SysFirstFile(szSpoolPath, filename);
		if (fh != SYS_INVALID_HANDLE)
		{
			do
			{
				if (SysIsDirectory(fh) || IsDotFilename(filename))
					continue;
				ISSQueueStatisticalData::IncrUnchecked(filename);
				SysMsSleep(100);
			} while (SysNextFile(fh, filename));
			SysFindClose(fh);
		}
	}
	if (dropdir)
	{
		SysFree(dropdir);
		dropdir= 0;
	}
	// check name of file to be used for data transfer to spam filter
	char * statfile= SvrGetConfigVar(hSvrConfig, "StatisticDataExchangeFile", "");
	if (statfile && *statfile)
	{
		strcpy(S_ExchangeFile, statfile);
	}
	if (statfile)
	{
		SysFree(statfile);
		statfile= 0;
	}
	SvrReleaseConfigHandle(hSvrConfig);
	while (S_thread)
	{
		S_thread->SendToProventia();
		SysMsSleep(2000);
	}
	return 0;
}

}	// namespace	ISSQueueStatisticalData
#endif	// ISS_EXTENSION_STATISTICS

#ifdef ISS_EXTENSION_UNCHECKEDDIR
static const char* TOUCH_FILE= ".modified";

size_t UncheckedDirectory::MAX_DIRS= 50;
size_t UncheckedDirectory::MAX_FILES = 250;

UncheckedDirectory::UncheckedDirectory()
: _init(false)
, _currentDir(INVALID_SLIST_PTR)
{
	_root[0]= 0;
	_mutex= SysCreateMutex();
	MutexAutoPtr locker(_mutex);
	ListInit(_directoryQueue);
	ISS_DEBUG_MSG(("ISS unchecked directory object created\n"));
}


int UncheckedDirectory::Initialize()
{
	if (_init)
		return ERR_SUCCESS;

	MutexAutoPtr locker(_mutex);

	try
	{
		if (_init)
		{
			return ERR_SUCCESS;
		}
		SVRCFG_HANDLE hSvrConfig = SvrGetConfigHandle();
		char *dropdir= SvrGetConfigVar(hSvrConfig, "SmtpSvrUncheckedDir");
		if (dropdir && *dropdir)
		{
			SvrGetSpoolDir(_root, sizeof(_root) );
			AppendSlash(_root);
			StrNCat(_root, dropdir, sizeof(_root));
			SysMakeDir(_root);
			AppendSlash(_root);
			ISS_DEBUG_MSG(("ISS unchecked directory working on directory %s\n", _root));
			SysFree(dropdir);
		}
		else
		{
			SysLogMessage(LOG_LEV_ERROR, "ISS unchecked directory working on empty root!\n");
			*_root= 0;
			_init= true;
			if (dropdir)
				SysFree(dropdir);
			return false;
		}

		ListPurgeFree(_directoryQueue);

		// create directory structure under unchecked
		if (*_root)
		{
			size_t i;
			for (i= 0; i<MAX_DIRS; ++i)
			{
				DirectoryInfo* temp= (DirectoryInfo*)SysAlloc(sizeof(DirectoryInfo));
				ListLinkInit(temp);
				temp->_fileCount= 0;
				strcpy(temp->_path, _root);

				// create subdirectory .00NNNN
				char buffer[16];
				sprintf(buffer, ".00%04d", i);
				strcat(temp->_path, buffer);

				// create touch file name to speed up touching procedure
				strcpy(temp->_touchPath, temp->_path);
				AppendSlash(temp->_touchPath);
				strcat(temp->_touchPath, TOUCH_FILE);

				// create directories 00NNNN, where NNNN is 0000 to MAX_DIRS
				bool exists= SysExistDir(temp->_path) != 0;
				if (!exists)
					exists= SysMakeDir(temp->_path) == 0;
				if (exists)
				{
					bool fileexists= SysExistFile(temp->_touchPath) != 0;
					if (!fileexists)
					{
#ifdef _WIN32
						HANDLE hFile = CreateFile(temp->_touchPath, GENERIC_WRITE,
										   FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
										   NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

						if (hFile != INVALID_HANDLE_VALUE)
							CloseHandle(hFile);
#else
						int fd= open(temp->_touchPath, O_CREAT | O_EXCL | O_RDWR, S_IREAD | S_IWRITE);
						if (fd > -1)
							close(fd);
#endif
					}
					ListAddTail(_directoryQueue, (PLISTLINK)temp); 
				}
				else
				{
					SysLogMessage(LOG_LEV_ERROR, "ISS unchecked directory could not create directory %s!\n",
							temp->_path);
					SysFree(temp);
				}
			}
			if (_directoryQueue == INVALID_SLIST_PTR)
			{
				SysLogMessage(LOG_LEV_ERROR, "ISS unchecked directory could not create any directory in %s!\n",
						_root);
				_init= true;
				ErrSetErrorCode(ERR_FILE_CREATE, "creating unchecked directory structure");
				return ERR_MEMORY;
			}
		}
		// scan all directories, set file count per directory
		char dummy[SYS_MAX_PATH];
		DirectoryInfo * dir;
		for (dir= (DirectoryInfo*)ListFirst(_directoryQueue); dir != INVALID_SLIST_PTR; 
				dir= (DirectoryInfo*)ListNext(_directoryQueue, (PLISTLINK)dir))
		{
			const char* currdir= dir->_path;
			SYS_HANDLE sh= SysFirstFile(currdir, dummy);
			if (sh == SYS_INVALID_HANDLE)
				break;
			do
			{
				if (SysIsDirectory(sh) || IsDotFilename(dummy))
					continue;
				++dir->_fileCount;
			} while (SysNextFile(sh, dummy));
			SysFindClose(sh);
			ISS_DEBUG_MSG(("ISS unchecked directory %s contains %d files\n",
				dir->_path, dir->_fileCount));
		}
	}
	catch(...)
	{
	}
	_init= true;
	return ERR_SUCCESS;
}

const char* UncheckedDirectory::GetSavePath()
{
	MutexAutoPtr locker(_mutex);

	const char * retval= _root;
	if (_directoryQueue == INVALID_SLIST_PTR)
		return retval;
	try
	{
		_currentDir= GetNextDirToUse();
		if (_currentDir != INVALID_SLIST_PTR)
		{
			retval= _currentDir->_path;
			ISS_DEBUG_MSG(("ISS unchecked directory continuing with %s\n",
				_currentDir->_path));
			++_currentDir->_fileCount;
		}
		else
		{
			SysLogMessage(LOG_LEV_ERROR, "ISS unchecked directory found no subdirectorie, defaulting to %s!\n",
				retval);
		}
		// touch the file TOUCH_FILE (".modified") in the returned folder
		if (_currentDir != INVALID_SLIST_PTR)
			SysSetFileModTime(_currentDir->_touchPath, time(0));
		else
		{
			char buffer[SYS_MAX_PATH]= "";
			strcpy(buffer, retval);
			AppendSlash(buffer);
			strcat(buffer, TOUCH_FILE);	
			SysSetFileModTime(buffer, time(0));
		}
	}
	catch(...)
	{
	}
	return retval;
}

UncheckedDirectory::DirectoryInfo * UncheckedDirectory::GetNextDirToUse()
{
	if (_currentDir == INVALID_SLIST_PTR)
	{
		// first call, we search the first with < MAX_FILES entry or the one with the minumum
		// of files
		DirectoryInfo * dir, *minPtr= INVALID_SLIST_PTR;
		for (dir= (DirectoryInfo*)ListFirst(_directoryQueue); dir != INVALID_SLIST_PTR; 
				dir= (DirectoryInfo*)ListNext(_directoryQueue, (PLISTLINK)dir))
		{
			if (dir->_fileCount<MAX_FILES)
				return dir;
			if (minPtr == INVALID_SLIST_PTR || minPtr->_fileCount > dir->_fileCount)
				minPtr= dir;
		}

		// reset if overflow
		if (minPtr->_fileCount >= MAX_FILES)
			minPtr->_fileCount= 0;
		return minPtr;
	}
	
	// if we have space left in current dir, reuse it
	if (_currentDir->_fileCount<MAX_FILES)
		return _currentDir;

	DirectoryInfo * dir= (DirectoryInfo*)ListNext(_directoryQueue, (PLISTLINK)_currentDir);
	// at end
	if (dir == INVALID_SLIST_PTR)
		dir = (DirectoryInfo*)ListFirst(_directoryQueue);
	// reset file counter to 0
	dir->_fileCount= 0;
	return dir;
}

UncheckedDirectory::~UncheckedDirectory()
{
	{
		MutexAutoPtr locker(_mutex);
		ListPurgeFree(_directoryQueue);
		_directoryQueue= 0;
	}
	SysCloseMutex(_mutex);
}

#ifdef ISS_EXTENSION_CTRL
int UncheckedDirectory::GetFileList(const char* tempFile)
{
	MutexAutoPtr locker(_mutex);
	int ret= Initialize();
	if (ret != ERR_SUCCESS)
		return ret;
	
	FILE *pListFile = fopen(tempFile, "wt");

	if (pListFile == NULL) {
		ErrSetErrorCode(ERR_FILE_CREATE, tempFile);
		return (ERR_FILE_CREATE);
	}

	char dummy[SYS_MAX_PATH];
	// search root directory first
	{
		SYS_HANDLE sh= SysFirstFile(_root, dummy);
		if (sh != SYS_INVALID_HANDLE)
			do
			{
				if (SysIsDirectory(sh) || IsDotFilename(dummy))
					continue;

				char szCurrPath[SYS_MAX_PATH];
				sprintf(szCurrPath, "%s%s", _root, dummy);
				
				SYS_FILE_INFO FI;
				if (SysGetFileInfo(szCurrPath, FI) < 0)
				{
					fclose(pListFile);
					return (ErrGetErrorCode());
				}
				if (DumpMessage(szCurrPath, pListFile) < 0)
				{
					fclose(pListFile);
					return ErrGetErrorCode();
				}
			} while (SysNextFile(sh, dummy));
		SysFindClose(sh);
	}

	DirectoryInfo * dir;
	for (dir= (DirectoryInfo*)ListFirst(_directoryQueue); dir != INVALID_SLIST_PTR; 
			dir= (DirectoryInfo*)ListNext(_directoryQueue, (PLISTLINK)dir))
	{
		const char* currdir= dir->_path;
		SYS_HANDLE sh= SysFirstFile(currdir, dummy);
		if (sh == SYS_INVALID_HANDLE)
			break;
		do
		{
			if (SysIsDirectory(sh) || IsDotFilename(dummy))
				continue;

			char szCurrPath[SYS_MAX_PATH];
			sprintf(szCurrPath, "%s%s%s", currdir, SYS_SLASH_STR, dummy);
			
			SYS_FILE_INFO FI;
			if (SysGetFileInfo(szCurrPath, FI) < 0)
			{
				fclose(pListFile);
				return (ErrGetErrorCode());
			}
			if (DumpMessage(szCurrPath, pListFile) < 0)
			{
				fclose(pListFile);
				return ErrGetErrorCode();
			}
		} while (SysNextFile(sh, dummy));
		SysFindClose(sh);
	}
	fclose(pListFile);
	return 0;
}
#endif	// ISS_EXTENSION_CTRL
#endif	// ISS_EXTENSION_UNCHECKEDDIR


#undef	ISS_DEBUG_MSG


// custom greeting message
char* GetSmtpGreetingMessage(SVRCFG_HANDLE cfg, char * msg, size_t buffsize)
{
	if (!msg || !buffsize)
		return 0;
	*msg= 0;
	// replace SMTP_SERVER_NAME in greeting and error message
	// read config string "SmtpSvrGreetingMessage"
	bool releaseConfig= false;
	if (cfg==INVALID_SVRCFG_HANDLE)
	{
		cfg= SvrGetConfigHandle();
		if (cfg == INVALID_SVRCFG_HANDLE)
			return msg;
		releaseConfig= true;
	}
	char * retval= SvrGetConfigVar(cfg, "SmtpSvrGreetingMessage");
	if (retval)
	{
		strncpy(msg, retval, buffsize-1);
		msg[buffsize-1]= 0;
		SysFree(retval);
	}
	if (releaseConfig)
		SvrReleaseConfigHandle(cfg);
	return msg;
}

}	// namespace ISSExtension
#endif	// ISS_EXTENSION
// </ISS EXTENSION>


